<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Image Updater
 * =============
 * Updater for JCOGS Image add-on
 * 
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

require_once PATH_THIRD . "jcogs_img/config.php";

class Jcogs_img_upd
{

    public function install()
    {
        $data = array(
            'module_name'      => JCOGS_IMG_CLASS,
            'module_version' => JCOGS_IMG_VERSION,
            'has_cp_backend' => 'y'
        );

        ee()->db->insert('modules', $data);

        // Register hooks
        $this->_register_hook('template_post_parse', 'template_post_parse');
        $this->_register_hook('after_file_save', 'after_file_update');
        $this->_register_hook('after_file_delete', 'after_file_update');
        $this->_register_hook('cp_custom_menu', 'cp_custom_menu');

        // Create settings db table
        ee()->load->dbforge();
        $fields = array(
            'id'                  =>  array('type' => 'int', 'constraint' => '10', 'unsigned' => true, 'null' => false, 'auto_increment' => true),
            'site_id'             => array('type' => 'INT'),
            'setting_name'        => array('type' => 'VARCHAR', 'constraint' => '100'),
            'value'               => array('type' => 'TEXT')
        );

        ee()->dbforge->add_field($fields);
        ee()->dbforge->add_key('id', TRUE);
        ee()->dbforge->create_table('jcogs_img_settings');

        ee('jcogs_img:Settings')->save_settings();

        // Clear the EE cache just for safety reasons
        ee()->cache->delete('/'.JCOGS_IMG_CLASS.'/');

        return true;
    }

    public function update($current = '')
    {
        // Update add-on version information
        ee('jcogs_img:Settings')::$settings['jcogs_add_on_class'] = JCOGS_IMG_CLASS;
        ee('jcogs_img:Settings')::$settings['jcogs_add_on_name'] = JCOGS_IMG_NAME;
        ee('jcogs_img:Settings')::$settings['jcogs_add_on_version'] = JCOGS_IMG_VERSION;

        // Resave settings to instantiate new parameters
        ee('jcogs_img:Settings')->save_settings();

        if (version_compare($current, '1.2', '<')) {
            // Register hooks
            $this->_register_hook('template_post_parse', 'template_post_parse');
        }

        if (version_compare($current, '1.2.11', '<')) {
            // Clear Image cache if updating from less than 1.2.11 (last cache hash breaking change)
            $cache_path = ee('jcogs_img:Utilities')->get_base_path() . ee('jcogs_img:Settings')::$settings['img_cp_default_cache_directory'];
            if (ee('Filesystem')->exists($cache_path)) {
                ee('Filesystem')->emptyDir($cache_path);
            }
            // Clear the EE cache just for safety reasons
            ee()->cache->delete('/'.JCOGS_IMG_CLASS.'/');
        }

        if (version_compare($current, '1.3', '<')) {
            // Clear the EE cache just for safety reasons
            ee()->cache->delete('/'.JCOGS_IMG_CLASS.'/');
        }

        if (version_compare($current, '1.3.3', '<')) {
            // Clear the EE cache just for safety reasons (due to Jumpmenu update)
            ee()->cache->delete('/'.JCOGS_IMG_CLASS.'/');
        }

        if (version_compare($current, '1.3.4', '<')) {
            // In case user has set value for PNG quality copy over to new variable
            if (array_key_exists('img_cp_jpg_default_png_quality', ee('jcogs_img:Settings')::$settings) && ee('jcogs_img:Settings')::$settings['img_cp_jpg_default_png_quality'] != '-1') {
                ee('jcogs_img:Settings')::$settings['img_cp_png_default_quality'] = ee('jcogs_img:Settings')::$settings['img_cp_jpg_default_png_quality'];
                // Resave settings to instantiate new parameters
                ee('jcogs_img:Settings')->save_settings();
            }
        }

        if (version_compare($current, '1.3.7', '<')) {
            // Register hooks
            $this->_register_hook('after_file_save', 'after_file_update');
            $this->_register_hook('after_file_delete', 'after_file_update');
        }

        if (version_compare($current, '1.3.9', '<') && (version_compare(APP_VER, '6.0.0', '>=')))
        {
            // Register hooks
            $this->_register_hook('cp_custom_menu', 'cp_custom_menu');
            // Clear Jump Menu cache to insert new text
            ee('CP/JumpMenu')->clearAllCaches();
        }

        if (version_compare($current, '1.3.20', '<'))
        {
            // Clear the EE cache just for safety reasons
            ee()->cache->delete('/'.JCOGS_IMG_CLASS.'/');
        }

        return true;
    }

    public function uninstall()
    {
        // Clear the EE caches just for safety reasons
        ee()->cache->delete('/'.JCOGS_IMG_CLASS.'/');

        // Clear the default image cache if setting exists
        $cache_path = ee('jcogs_img:Utilities')->get_base_path() . ee('jcogs_img:Settings')::$settings['img_cp_default_cache_directory'];
        if (ee('Filesystem')->exists($cache_path)) {
            ee('Filesystem')->emptyDir($cache_path);
        }

        //delete the module
        ee()->db->where('module_name', JCOGS_IMG_CLASS);
        ee()->db->delete('modules');

        //remove actions
        ee()->db->where('class', JCOGS_IMG_CLASS);
        ee()->db->delete('actions');

        //remove extensions
        ee()->db->where('class', JCOGS_IMG_CLASS . '_ext');
        ee()->db->delete('extensions');

        ee()->load->dbforge();
        ee()->db->where('module_name', 'jcogs_img');
        ee()->db->delete('modules');

        //Remove the settings table from the database
        ee()->dbforge->drop_table('jcogs_img_settings');

        return true;
    }

    private function _register_hook($hook, $method = NULL, $priority = 10)
    {
        if (is_null($method)) {
            $method = $hook;
        }

        if (
            ee()->db->where('class', JCOGS_IMG_CLASS)
            ->where('hook', $hook)
            ->count_all_results('extensions') == 0
        ) {
            ee()->db->insert('extensions', array(
                'class'        => JCOGS_IMG_CLASS . '_ext',
                'method'    => $method,
                'hook'        => $hook,
                'settings'    => '',
                'priority'    => $priority,
                'version'    => JCOGS_IMG_VERSION,
                'enabled'    => 'y'
            ));
        }
    }
}
