<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Image Master Control Panel
 * ==========================
 * Defines the CP pages used by the  add-on
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

require_once PATH_THIRD . "jcogs_img/vendor/autoload.php";
require_once PATH_THIRD . "jcogs_img/config.php";

ee()->lang->load('jcogs_lic', ee()->session->get_language(), false, true, PATH_THIRD . 'jcogs_img/');

class Jcogs_img_mcp
{
    private $_settings = array();
    private $_data = array();
    private $license_status = null;
    private $license_key_email = '';
    private $cache_path_ok = false;
    private $cache_path = '';

    public function __construct()
    {
        $this->_settings = ee('jcogs_img:Settings')::$settings;

        // Check we have a license in place
        if (!($this->_settings['jcogs_license_mode'] == 'valid' || $this->_settings['jcogs_license_mode'] == 'magic')) {
            // Check to see if we can run in demo mode
            $this->_settings['jcogs_license_mode'] = ee('jcogs_img:Licensing')->is_demo_mode_allowed();
        }

        // Check php version is valid... 
        if (!ee('jcogs_img:Utilities')->valid_php_version()) {
            // We cannot operate without a valid php version so bale... 
            ee('CP/Alert')->makeBanner('jcogs_img_cp_php_version_invalid')
                ->asIssue()
                ->withTitle(lang('jcogs_img_cp_php_version_invalid'))
                ->addToBody(sprintf(lang('jcogs_img_cp_php_version_invalid_desc'), strval(PHP_VERSION)))
                ->now();
        }

        // Check EE version is valid... 
        if (!ee('jcogs_img:Utilities')->valid_ee_version()) {
            // We cannot operate without a valid php version so bale... 
            ee('CP/Alert')->makeBanner('jcogs_img_cp_ee_version_invalid')
                ->asIssue()
                ->withTitle(lang('jcogs_img_cp_ee_version_invalid'))
                ->addToBody(sprintf(lang('jcogs_img_cp_ee_version_invalid_desc'), strval(APP_VER)))
                ->now();
        }

        // Check we have a valid cache path in place
        if (!$this->cache_path_ok) {
            // Get the path to the cache
            $this->cache_path = $this->_settings['img_cp_default_cache_directory'];
            // Is it a valid cache path?
            $this->cache_path_ok = ee('jcogs_img:Utilities')->path($this->cache_path, true) ? true : false;

            if (!$this->cache_path_ok) {
                // We cannot operate without a cache path so bale... 
                ee('CP/Alert')->makeBanner('jcogs_img_cp_cache_path_invalid')
                    ->asIssue()
                    ->withTitle(lang('jcogs_img_cp_cache_path_invalid'))
                    ->addToBody(lang('jcogs_img_cp_cache_path_invalid_desc'))
                    ->now();
            }
        }
    }

    public function index()
    {
        ee()->load->library('file_field');
        ee()->load->helper('url');
        $this->_build_sidebar();
        ee()->cp->add_js_script(array(
            'file' => array('cp/form_group'),
        ));

        // --------------------------------------
        // Validate and then save any changes
        // --------------------------------------
        if ($_POST) {

            // Validation
            $validator = ee('Validation')->make();

            // Define custom validation rules
            // ------------------------------

            // 1) Valid php version
            // --------------------
            $validator->defineRule('valid_php_version', function ($key, $value, $parameters) {
                // Add-on will only work when installed on systems with php 7.3 or better.
                if (!ee('jcogs_img:Utilities')->valid_php_version()) {
                    return 'jcogs_img_cp_php_version_invalid';
                }
                return true;
            });

            // 2) Valid EE version
            // --------------------
            $validator->defineRule('valid_ee_version', function ($key, $value, $parameters) {
                // Add-on will only work when installed on systems with php 7.4 or better.
                if (!ee('jcogs_img:Utilities')->valid_ee_version()) {
                    return 'jcogs_img_cp_ee_version_invalid';
                }
                return true;
            });

            $validator->setRules(array(
                'enable_img'                     => 'enum[y,n]|valid_php_version|valid_ee_version',
                'img_cp_class_always_output_full_urls' => 'enum[y,n]',
                'img_cp_enable_debugging'        => 'enum[y,n]',
            ));

            $result = $validator->validate($_POST);
            $extra_message_line = '';
            if (isset($this->license_status) && property_exists($this->license_status, 'message')) {
                $extra_message_line = $this->license_status->message;
            }

            if ($result->isValid()) {

                $fields = array();
                // Get all $_POST values, store them in array and save them
                // Use ee input library as it cleans up POST entries on loading
                foreach ($_POST as $key => $value) {
                    $fields[$key] = ee()->input->post($key);
                    if ($key == 'img_cp_default_cache_directory') {
                        // Remove trailing slash if there is one.
                        $fields[$key] = rtrim($fields[$key], '/');
                    }
                    $fields[$key] = is_numeric($fields[$key]) ? (int) $fields[$key] : $fields[$key];
                }

                $fields = array_merge($this->_settings, $fields);

                // Fix obscured license field if we need to
                if ($fields['jcogs_license_key'] == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    $fields['jcogs_license_key'] = $this->_settings['jcogs_license_key'];
                }

                // Now save the settings values
                ee('jcogs_img:Settings')->save_settings($fields);

                // Pop up a save confirmation if all went well.
                ee('CP/Alert')->makeInline('shared-form')
                    ->asSuccess()
                    ->withTitle(lang('preferences_updated'))
                    ->addToBody(lang('preferences_updated_desc'))
                    ->addToBody($extra_message_line)
                    ->defer();

                // Redraw page now
                ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img')->compile());
            } else {
                $this->_data['errors'] = $result;
                ee('CP/Alert')->makeInline('shared-form')
                    ->asIssue()
                    ->withTitle(lang('settings_save_error'))
                    ->addToBody(lang('settings_save_error_desc'))
                    ->addToBody($extra_message_line)
                    ->now();
            }
        }

        // No post data, so just draw the page

        // --------------------------------------
        // Build the form into $sections array
        // --------------------------------------

        $sections = array();

        $sections[lang('jcogs_img_cp_system_enable')] = array(
            'group' => 'license_options',
            'settings' => array(

                // ----------------------------------------
                // Global on/off switch
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_globally',
                    'desc' => 'jcogs_img_cp_enable_globally_desc',
                    'fields' => array(
                        'enable_img' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['enable_img'],
                            'group_toggle' => array(
                                'y' => 'img_options|path_option|cache_options|adv_options'
                            )
                        )
                    )
                )
            )
        );

        $sections[lang('jcogs_img_system_options_storage')] = array(
            'group' => 'path_options',
            'settings' => array(

                // ----------------------------------------
                // Always output full URLs
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_class_always_output_full_urls',
                    'desc' => 'jcogs_img_cp_class_always_output_full_urls_desc',
                    'fields' => array(
                        'img_cp_class_always_output_full_urls' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_class_always_output_full_urls'],
                            'group_toggle' => array(
                                'n' => 'prefix_options'
                            )
                        )
                    )
                ),

                // ----------------------------------------
                // Set CDN remote path prefix
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_path_prefix',
                    'desc' => 'jcogs_img_cp_path_prefix_desc',
                    'group' => 'prefix_options',
                    'fields' => array(
                        'img_cp_path_prefix' => array(
                            'type'  => 'text',
                            'value' => trim($this->_settings['img_cp_path_prefix']),
                            'required' => false
                        )
                    )
                ),
            )
        );

        $sections[lang('jcogs_img_system_options_debug')] = array(
            'group' => 'app_options',
            'settings' => array(
                // --------------------------------------
                // Enable debugging reports?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_debugging',
                    'desc' => 'jcogs_img_cp_enable_debugging_desc',
                    'fields' => array(
                        'img_cp_enable_debugging' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_debugging']
                        )
                    )
                )
            )
        );

        $this->_data += array(
            'cp_page_title' => lang('jcogs_img_system_options'),
            'base_url' => ee('CP/URL', 'addons/settings/jcogs_img')->compile(),
            'save_btn_text' => sprintf(lang('btn_save'), lang('jcogs_img_cp_main_settings')),
            'save_btn_text_working' => lang('btn_saving'),
            'sections' => $sections
        );

        return array(
            'heading'       => lang('jcogs_img_cp_main_settings'),
            'breadcrumb'    => array(
                ee('CP/URL', 'addons/settings/jcogs_img')->compile() => lang('jcogs_img_module_name')
            ),
            'body'          => ee('View')->make('ee:_shared/form')->render($this->_data),
        );
    }

    public function advanced_settings()
    {
        ee()->load->library('file_field');
        $this->_build_sidebar();
        ee()->cp->add_js_script(array(
            'file' => array('cp/form_group'),
        ));

        // --------------------------------------
        // Validate and then save any changes
        // --------------------------------------
        if ($_POST) {

            // Validation
            $validator = ee('Validation')->make();

            // Define custom validation rules
            // ------------------------------

            // 1) Valid user agent string
            // --------------------------
            $validator->defineRule('valid_user_agent_string', function ($key, $value, $parameters) {
                // Not a good regex, but if it fails this it probably is not a user agent string... 
                $browser_info = ee('jcogs_img:Utilities')->getBrowser($value);
                if (!$browser_info['userAgent'] || !$browser_info['platform']) {
                    return 'jcogs_img_cp_user_agent_string';
                }
                return true;
            });

            // 2) Valid CE Image Remote Directory
            // ----------------------------------
            // If not equal to default value, needs to be valid directory
            $validator->defineRule('valid_ce_image_remote_dir', function ($key, $value, $parameters) {
                if ($value != trim($this->_settings['img_cp_ce_image_remote_dir']) && !ee('Filesystem')->exists($value)) {
                    return 'jcogs_img_cp_valid_ce_image_remote_dir';
                }
                return true;
            });

            // 3) Valid filename separator
            // ---------------------------
            $validator->defineRule('valid_separator', function ($key, $value, $parameters) {
                // Check the length and for spaces and for reserved characters
                if (
                    !strlen($value) ||
                    preg_match('/\s+/', $value) ||
                    preg_match('/[\[\^\-\\\]\_\.\~\!\*\'\(\)\;\:\@\&\=\+\$\,\/\?\%\#]/', $value)
                ) {
                    return  'jcogs_img_cp_invalid_separator_string';
                }
                return true;
            });

            // 4) Valid licensing domain
            // -------------------------
            $validator->defineRule('valid_licensing_domain', function ($key, $value, $parameters) {
                // Find out if we can poll the licensing server... 
                $action_array = @file_get_contents('https://' . $value . '/actions');
                if ($action_array != false) {
                    return true;
                }
                return  'jcogs_lic_cp_invalid_licensing_domain';
            });


            $validator->setRules(array(
                'img_cp_enable_browser_check' => 'enum[y,n]',
                'img_cp_class_consolidation_default' => 'enum[y,n]',
                'img_cp_default_filename_separator' => 'valid_separator|required',
                'img_cp_ce_image_remote_dir' => 'whenPresent|valid_ce_image_remote_dir',
                'img_cp_default_max_source_filename_length' => 'integer|greaterThan[0]|lessThan[201]|required',
                'img_cp_default_min_php_ram' => 'integer|greaterThan[0]|required',
                'img_cp_default_min_php_process_time' => 'integer|greaterThan[-1]|required',
                'img_cp_default_php_remote_connect_time' => 'integer|greaterThan[0]|required',
                'img_cp_default_user_agent_string' => 'valid_user_agent_string|required',
                'jcogs_license_server_domain'    => 'valid_licensing_domain'
            ));

            $result = $validator->validate($_POST);

            if ($result->isValid()) {

                $fields = array();
                // Get all $_POST values, store them in array and save them
                // Use ee input library as it cleans up POST entries on loading
                foreach ($_POST as $key => $value) {
                    $fields[$key] = ee()->input->post($key);
                    if ($key == 'img_cp_default_cache_directory') {
                        // Remove trailing slash if there is one.
                        $fields[$key] = rtrim($fields[$key], '/');
                    }
                    $fields[$key] = is_numeric($fields[$key]) ? (int) $fields[$key] : $fields[$key];
                }

                $fields = array_merge($this->_settings, $fields);

                // Fix obscured license field if we need to
                if ($fields['jcogs_license_key'] == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    $fields['jcogs_license_key'] = $this->_settings['jcogs_license_key'];
                }

                // Now save the settings values
                ee('jcogs_img:Settings')->save_settings($fields);

                // Pop up a save confirmation if all went well.
                ee('CP/Alert')->makeInline('shared-form')
                    ->asSuccess()
                    ->withTitle(lang('preferences_updated'))
                    ->addToBody(lang('preferences_updated_desc'))
                    ->defer();

                // Redraw page now
                ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img/advanced_settings')->compile());
            } else {
                $this->_data['errors'] = $result;
                ee('CP/Alert')->makeInline('shared-form')
                    ->asIssue()
                    ->withTitle(lang('settings_save_error'))
                    ->addToBody(lang('settings_save_error_desc'))
                    ->now();
            }
        }

        // No post data, so just draw the page

        // --------------------------------------
        // Build the form into $sections array
        // --------------------------------------

        // --------------------------------------
        // Enable JCOGS Image Advanced options?
        // --------------------------------------

        $sections[lang('jcogs_img_advanced_settings')] = array(
            'group' => 'adv_options',
            'settings' => array(
                array(
                    'title' => 'jcogs_img_cp_advanced_options',
                    'desc' => 'jcogs_img_cp_advanced_options_desc',
                    'fields' => array(
                        'img_cp_advanced_options' => array(
                            'type'  => 'yes_no',
                            'value' => 0,
                            'group_toggle' => array(
                                'y' => 'advanced_options'
                            )
                        )
                    )
                ),

                // ----------------------------------------
                // Enable browser capability detection
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_browser_check',
                    'desc' => 'jcogs_img_cp_enable_browser_check_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_enable_browser_check' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_browser_check']
                        )
                    )
                ),

                // ----------------------------------------
                // Enable class consolidation capability
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_class_consolidation_default',
                    'desc' => 'jcogs_img_cp_class_consolidation_default_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_class_consolidation_default' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_class_consolidation_default']
                        )
                    )
                ),

                // ----------------------------------------
                // Enable attribute variable expansion capability
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_attribute_variable_expansion_default',
                    'desc' => 'jcogs_img_cp_attribute_variable_expansion_default_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_attribute_variable_expansion_default' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_attribute_variable_expansion_default']
                        )
                    )
                ),

                // ----------------------------------------
                // Set filename separator
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_filename_separator',
                    'desc' => 'jcogs_img_cp_default_filename_separator_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_default_filename_separator' => array(
                            'type'  => 'text',
                            'value' => trim($this->_settings['img_cp_default_filename_separator']),
                            'required' => true
                        )
                    )
                ),

                // ------------------------------------------
                // Set default max length for source filename
                // ------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_max_source_filename_length',
                    'desc' => 'jcogs_img_cp_default_max_source_filename_length_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_default_max_source_filename_length' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_max_source_filename_length'],
                            'required' => true
                        )
                    )
                ),

                // ----------------------------------------
                // Set CE Image Remote Image Cache Directory
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_ce_image_remote_dir',
                    'desc' => 'jcogs_img_cp_ce_image_remote_dir_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_ce_image_remote_dir' => array(
                            'type'  => 'text',
                            'value' => trim($this->_settings['img_cp_ce_image_remote_dir']),
                            'required' => false
                        )
                    )
                ),

                // -------------------------------------------------
                // Set php memory limit to request during operation
                // -------------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_min_php_ram',
                    'desc' => 'jcogs_img_cp_default_min_php_ram_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_default_min_php_ram' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_min_php_ram'],
                            'required' => true
                        )
                    )
                ),

                // --------------------------------------------------------
                // Set php execution time limit to request during operation
                // --------------------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_min_php_process_time',
                    'desc' => 'jcogs_img_cp_default_min_php_process_time_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_default_min_php_process_time' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_min_php_process_time'],
                            'required' => true
                        )
                    )
                ),

                // --------------------------------------------------------
                // Set php connection time limit for remote file retrieval
                // --------------------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_php_remote_connect_time',
                    'desc' => 'jcogs_img_cp_default_php_remote_connect_time_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_default_php_remote_connect_time' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_php_remote_connect_time'],
                            'required' => true
                        )
                    )
                ),

                // --------------------------------------------------------
                // Set user agent string to use for remote file retrieval
                // --------------------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_user_agent_string',
                    'desc' => 'jcogs_img_cp_default_user_agent_string_desc',
                    'group' => 'advanced_options',
                    'fields' => array(
                        'img_cp_default_user_agent_string' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_user_agent_string'],
                            'required' => true
                        )
                    )
                ),

                // --------------------------------------------------------
                // Location of JCOGS Licensing server
                // --------------------------------------------------------

                ee('jcogs_img:Licensing')->mcp_licensing_server_domain_entry($this->_settings['jcogs_license_server_domain']),

            )
        );

        $this->_data += array(
            'cp_page_title' => lang('jcogs_img_advanced_settings_heading'),
            'base_url' => ee('CP/URL', 'addons/settings/jcogs_img/advanced_settings')->compile(),
            'save_btn_text' => sprintf(lang('btn_save'), lang('jcogs_img_advanced_settings')),
            'save_btn_text_working' => lang('btn_saving'),
            'sections' => $sections
        );

        // Tell EE to load the custom javascript for the page
        // ee()->cp->load_package_js('form_controls');

        return array(
            'heading'       => lang('jcogs_img_advanced_settings'),
            'breadcrumb'    => array(
                ee('CP/URL', 'addons/settings/jcogs_img/advanced_settings')->compile() => lang('jcogs_img_module_name')
            ),
            'body'          => ee('View')->make('ee:_shared/form')->render($this->_data),
        );
    }

    public function caching()
    {
        ee()->load->library('file_field');
        ee()->load->helper('url');
        $this->_build_sidebar();
        ee()->cp->add_js_script(array(
            'file' => array('cp/form_group'),
        ));

        // --------------------------------------
        // Validate and then save any changes
        // --------------------------------------
        if ($_POST) {

            // Validation
            $validator = ee('Validation')->make();

            // Define custom validation rules
            // ------------------------------

            // 1) Valid cache path
            // -------------------
            $validator->defineRule('valid_cache_path', function ($key, $value, $parameters) {

                // Cache directory needs to be included in URLs, so needs to be valid url element
                // So we need to check for non valid elements (e.g. spaces etc.)

                // First parse the stub given
                $parsed_url = parse_url($value);

                // Check to see that we only got a path value back (that should be only element provided)
                if ((!$parsed_url || count($parsed_url) > 1) && isset($parsed_url['path'])) {
                    // Something seriously wrong with provided value so bale
                    return 'jcogs_img_cp_invalid_url_path';
                }

                // Now strip out any leading '/' or '//' elements
                ee()->load->helper('string');
                $parsed_url['path'] = ltrim(reduce_double_slashes($parsed_url['path']), '/');

                // Now check the path element plus site_url is valid URL
                // filter_var is flaky with non-standard domain names, so if it fails, do parse_url and see if we get at least three components when we do parse_url...
                // scheme, host, path... 
                if (!filter_var(base_url() . $parsed_url['path'], FILTER_VALIDATE_URL)) {
                    if (count(parse_url(base_url() . $parsed_url['path'])) < 3) {
                        return 'jcogs_img_cp_invalid_url_path';
                    }
                }

                // Now check we can create the directory

                // Get base path
                $base_path = ee('jcogs_img:Utilities')->get_base_path();
                if (!$base_path) {
                    // $basepath is invalid, so bale
                    return 'jcogs_cp_ee_base_path_invalid';
                }

                // Build the path to the directory suggested
                $path = $base_path . $value;

                // Check this could be a valid cache path ultimately by trying to create it
                return ee('Filesystem')->exists($path) || ee('Filesystem')->mkDir($path);
            });

            $validator->setRules(array(
                'img_cp_default_cache_directory'   => 'xss|valid_cache_path|required',
                'img_cp_path_prefix'               => 'whenPresent|url',
                'img_cp_cache_auto_manage'         => 'enum[y,n]',
                'img_cp_enable_cache_audit'        => 'enum[y,n]',
                'img_cp_default_cache_duration'    => 'integer|greaterThan[-2]|required',
                'img_cp_default_cache_audit_after' => 'integer|greaterThan[0]|required',
            ));

            $result = $validator->validate($_POST);
            $extra_message_line = '';
            if (isset($this->license_status) && property_exists($this->license_status, 'message')) {
                $extra_message_line = $this->license_status->message;
            }

            if ($result->isValid()) {

                $fields = array();
                // Get all $_POST values, store them in array and save them
                // Use ee input library as it cleans up POST entries on loading
                foreach ($_POST as $key => $value) {
                    $fields[$key] = ee()->input->post($key);
                    if ($key == 'img_cp_default_cache_directory') {
                        // Remove double slashes, leading and trailing slash if there is one.
                        ee()->load->helper('string');
                        $fields[$key] = trim(reduce_double_slashes($fields[$key]), '/');
                    }
                    $fields[$key] = is_numeric($fields[$key]) ? (int) $fields[$key] : $fields[$key];
                }

                $fields = array_merge($this->_settings, $fields);

                // Fix obscured license field if we need to
                if ($fields['jcogs_license_key'] == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    $fields['jcogs_license_key'] = $this->_settings['jcogs_license_key'];
                }

                // Now save the settings values
                ee('jcogs_img:Settings')->save_settings($fields);

                // Pop up a save confirmation if all went well.
                ee('CP/Alert')->makeInline('shared-form')
                    ->asSuccess()
                    ->withTitle(lang('preferences_updated'))
                    ->addToBody(lang('preferences_updated_desc'))
                    ->addToBody($extra_message_line)
                    ->defer();

                // Redraw page now
                ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img/caching')->compile());
            } else {
                $this->_data['errors'] = $result;
                ee('CP/Alert')->makeInline('shared-form')
                    ->asIssue()
                    ->withTitle(lang('settings_save_error'))
                    ->addToBody(lang('settings_save_error_desc'))
                    ->addToBody($extra_message_line)
                    ->now();
            }
        }

        // No post data, so just draw the page

        // Get the cache_audit info
        // Do we have a marker from last audit?
        $marker = ee('jcogs_img:Utilities')->cache_utility('get', '/' . JCOGS_IMG_CLASS . '/' . 'image_cache_audit');
        $last_audit = $marker ? $marker : 0;

        // When is next audit due?
        $next_audit = $last_audit + ee('jcogs_img:Settings')::$settings['img_cp_default_cache_audit_after'];

        // Build next audit message / button combo
        $cache_audit_button_block = '';

        // Don't show anything if Cache Audit is disabled
        if (substr(strtolower(ee('jcogs_img:Settings')::$settings['img_cp_enable_cache_audit']), 0, 1) == 'y' && (ee('jcogs_img:Settings')::$settings['img_cp_default_cache_audit_after'] > 0)) {
            // Is Audit due more than a day into the future?
            if ($next_audit - time() >  60 * 60 * 24) {
                $next_audit_message = sprintf(lang('jcogs_img_cp_cache_audit_now_since_days'), round(($next_audit - time()) / (60 * 60 * 24), 0));
            } else {
                $next_audit_message = lang('jcogs_img_cp_cache_audit_now_since_today');
            }

            // Build the message for cache audit button
            if ($this->_get_image_cache_info()['number_of_cache_fragments'] != '') {
                $audit_button_message = lang('jcogs_img_cp_cache_audit_now_desc');
            } else {
                $audit_button_message = lang('jcogs_img_cp_cache_audit_now_desc_empty');
            }
            $audit_button_message .= PHP_EOL .
                sprintf(lang('jcogs_img_cp_cache_audit_now_button'), ee('CP/URL', 'addons/settings/jcogs_img/manual_cache_audit'), $this->_get_image_cache_info()['number_of_cache_fragments'] == '' ? 'disabled' : '');

            // Build the actual entry for CP display
            $cache_audit_button_block = array(
                'title' => 'jcogs_img_cp_cache_audit_now',
                'desc' => $next_audit_message . ' ' . $audit_button_message,
                'group' => 'cache_audit',
                'fields' => array(
                    'img_cp_ee5_empty_fields_hack_3' => array(
                        'type'  => 'hidden',
                        'value' => '',
                        'required' => false
                    )
                )
            );
        }

        // --------------------------------------
        // Build the form into $sections array
        // --------------------------------------

        $sections = array();

        $sections['jcogs_img_cp_cache_status'] = array(
            'group' => 'cache_status',
            'settings' => array(

                // ----------------------------------------
                // Caching Status
                // ----------------------------------------

                $this->_get_image_cache_info()['cache_performance_desc'],

            )
        );

        $sections['jcogs_img_cp_cache_controls'] = array(
            'group' => 'cache_options',
            'settings' => array(

                // ----------------------------------------
                // Run manual Cache Audit
                // ----------------------------------------

                $cache_audit_button_block,

                // ----------------------------------------
                // Clear Cache
                // ----------------------------------------

                $this->_get_image_cache_info()['cache_clear_button_desc'],
            )
        );

        $sections['jcogs_img_cp_cache_settings'] = array(
            'group' => 'cache_options',
            'settings' => array(

                // --------------------------------------
                // Enable cache auto-manage?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_cache_auto_manage',
                    'desc' => 'jcogs_img_cp_enable_cache_auto_manage_desc',
                    'fields' => array(
                        'img_cp_cache_auto_manage' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_cache_auto_manage']
                        )
                    )
                ),

                // --------------------------------------
                // Enable cache audit?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_cache_audit',
                    'desc' => 'jcogs_img_cp_enable_cache_audit_desc',
                    'fields' => array(
                        'img_cp_enable_cache_audit' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_cache_audit'],
                            'group_toggle' => array(
                                'y' => 'cache_audit'
                            )
                        )
                    )
                ),

                // ----------------------------------------
                // Set default cache audit pause duration
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_cache_audit_after',
                    'desc' => 'jcogs_img_cp_default_cache_audit_after_desc',
                    'group' => 'cache_audit',
                    'fields' => array(
                        'img_cp_default_cache_audit_after' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_cache_audit_after'],
                            'required' => true
                        )
                    )
                ),

                // ----------------------------------------
                // Set default cache directory
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_cache_directory',
                    'desc' => 'jcogs_img_cp_default_cache_directory_desc',
                    'fields' => array(
                        'img_cp_default_cache_directory' => array(
                            'type'  => 'text',
                            'value' => $this->cache_path,
                            'preload' => lang('jcogs_img_cp_default_cache_directory_placeholder'),
                            'required' => true
                        )
                    )
                ),

                // ----------------------------------------
                // Set default cache duration
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_cache_duration',
                    'desc' => 'jcogs_img_cp_default_cache_duration_desc',
                    'fields' => array(
                        'img_cp_default_cache_duration' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_cache_duration'],
                            'preload' => lang('jcogs_img_cp_default_cache_duration_minus_one_option'),
                        )
                    )
                )
            )
        );

        $this->_data += array(
            'cp_page_title' => lang('jcogs_img_caching_page_title'),
            'base_url' => ee('CP/URL', 'addons/settings/jcogs_img/caching')->compile(),
            'save_btn_text' => sprintf(lang('btn_save'), lang('jcogs_img_cp_cache_settings')),
            'save_btn_text_working' => lang('btn_saving'),
            'sections' => $sections
        );

        return array(
            'heading'       => lang('jcogs_img_cp_cache_settings'),
            'breadcrumb'    => array(
                ee('CP/URL', 'addons/settings/jcogs_img/caching')->compile() => lang('jcogs_img_module_name')
            ),
            'body'          => ee('View')->make('ee:_shared/form')->render($this->_data),
        );
    }

    public function image_defaults()
    {
        ee()->load->library('file_field');
        $this->_build_sidebar();
        ee()->cp->add_js_script(array(
            'file' => array('cp/form_group'),
        ));

        // Build list of valid image formats for this server
        if (!$valid_server_image_options = ee('jcogs_img:Utilities')->cache_utility('get', '/' . JCOGS_IMG_CLASS . '/' . 'server_image_format_options')) {
            if (extension_loaded('gd')) {
                $server_gd_info = gd_info();
                // Work out what capabilities we have... 
                $valid_server_image_options = ['source' => 'Use format of source image',];
                foreach ($server_gd_info as $key => $value) {
                    if (!in_array(strtolower(substr($key, 0, 2)), ['gd', 'fr', 'ji'])) {
                        $this_capability = explode(' ', $key);
                        if ($value === true && strtolower($this_capability[1]) != 'read') {
                            $this_capability[0] = strtolower($this_capability[0]) != 'jpeg' ? $this_capability[0] : 'JPG';
                            $valid_server_image_options[strtolower($this_capability[0])] = $this_capability[0];
                        }
                    }
                }
            } else {
                $valid_server_image_options = ['source' => 'GD Library not found',];
            }
            ee('jcogs_img:Utilities')->cache_utility('save', JCOGS_IMG_CLASS . '/' . 'server_image_format_options', $valid_server_image_options, 60 * 60 * 24);
        }

        // --------------------------------------
        // Validate and then save any changes
        // --------------------------------------
        if ($_POST) {

            // Validation
            $validator = ee('Validation')->make();

            // Define custom validation rules
            // ------------------------------

            // 1) Valid remote image
            // ---------------------
            $validator->defineRule('valid_remote_image', function ($key, $value, $parameters) {
                // We are checking to see if URL given translates to an image file
                // We only get here if entry passes XSS and 'is a url' checks
                return $this->_valid_remote_image($value);
            });

            // 2) Conditional validation for when Local Fallback is selected
            // -------------------------------------------------------------
            // mirror $_POST to $data since $_POST can't be used in use statement for function calls
            $data = $_POST;

            $validator->defineRule('whenFallbackTypeIsLocal', function ($key, $value, $parameters, $rule) use ($data) {
                return ($data['img_cp_enable_default_fallback_image'] == 'yl') ? TRUE : $rule->skip();
            });

            // 3) Conditional validation for when Remote Fallback is selected
            // --------------------------------------------------------------
            $validator->defineRule('whenFallbackTypeIsRemote', function ($key, $value, $parameters, $rule) use ($data) {
                return ($data['img_cp_enable_default_fallback_image'] == 'yr') ? TRUE : $rule->skip();
            });

            // Set validation rules
            // --------------------
            $validator->setRules(array(
                'img_cp_default_max_image_dimension' => 'integer|greaterThan[-1]|required',
                'img_cp_default_img_width' => 'integer|greaterThan[-1]|required',
                'img_cp_default_img_height' => 'integer|greaterThan[-1]|required',
                'img_cp_default_max_image_size' => 'integer|greaterThan[0]|required'
            ));

            // Do the validation
            // -----------------
            $result = $validator->validate($_POST);

            if ($result->isValid()) {

                $fields = array();
                // Get all $_POST values, store them in array and save them
                // Use ee input library as it cleans up POST entries on loading
                foreach ($_POST as $key => $value) {
                    $fields[$key] = ee()->input->post($key);
                    if ($key == 'img_cp_default_cache_directory') {
                        // Remove trailing slash if there is one.
                        $fields[$key] = stripos($fields[$key], '/', -1) ? substr($fields[$key], 0, -1) : $fields[$key];
                    }
                    $fields[$key] = is_numeric($fields[$key]) ? (int) $fields[$key] : $fields[$key];
                }

                $fields = array_merge($this->_settings, $fields);

                // Fix obscured license field if we need to
                if ($fields['jcogs_license_key'] == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    $fields['jcogs_license_key'] = $this->_settings['jcogs_license_key'];
                }

                // Now save the settings values
                ee('jcogs_img:Settings')->save_settings($fields);

                // Pop up a save confirmation if all went well.
                ee('CP/Alert')->makeInline('shared-form')
                    ->asSuccess()
                    ->withTitle(lang('preferences_updated'))
                    ->addToBody(lang('preferences_updated_desc'))
                    ->defer();

                // Redraw page now
                ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img/image_defaults')->compile());
            } else {
                $this->_data['errors'] = $result;
                ee('CP/Alert')->makeInline('shared-form')
                    ->asIssue()
                    ->withTitle(lang('settings_save_error'))
                    ->addToBody(lang('settings_save_error_desc'))
                    ->now();
            }
        }

        // No post data, so just draw the page

        // --------------------------------------
        // Build the form into $sections array
        // --------------------------------------

        $sections[lang('jcogs_img_image_options_format_section')] = array(
            'group' => 'img_options',
            'settings' => array(

                // ----------------------------------------
                // Set default image format
                // ----------------------------------------

                array(
                    'title' => 'img_cp_default_image_format',
                    'desc' => 'img_cp_default_image_format_desc',
                    'fields' => array(
                        'img_cp_default_image_format' => array(
                            'type' => 'select',
                            'choices' => $valid_server_image_options,
                            'value' => $this->_settings['img_cp_default_image_format'],
                        )
                    )
                ),

                // ----------------------------------------
                // Set default image quality for jpgs
                // ----------------------------------------

                array(
                    'title' => 'img_cp_jpg_default_quality',
                    'desc' => 'img_cp_jpg_default_quality_desc',
                    'fields' => array(
                        'img_cp_jpg_default_quality' => array(
                            'type'  => 'html',
                            'content' => '<div style=\'display:flex;align-items:center;\'><div style=\'padding-right:0.5em;\'>0</div><input type=\'range\' min=\'0\' max=\'100\' step=\'1\' id=\'img_cp_jpg_default_quality\' name=\'img_cp_jpg_default_quality\' value=\'' . $this->_settings['img_cp_jpg_default_quality'] . '\' style=\'max-width:30vw;\' required ><div style=\'padding-left:0.5em;padding-right:0.5em;\'>100 - Current value: </div><div name=\'jcogs_dqs\'>' . $this->_settings['img_cp_jpg_default_quality'] . '</div></div>'
                        )
                    )
                ),

                // ----------------------------------------
                // Set default image quality for pngs
                // ----------------------------------------

                array(
                    'title' => 'img_cp_png_default_quality',
                    'desc' => 'img_cp_png_default_quality_desc',
                    'fields' => array(
                        'img_cp_png_default_quality' => array(
                            'type'  => 'html',
                            'content' => '<div style=\'display:flex;align-items:center;\'><div style=\'padding-right:0.5em;\'>0</div><input type=\'range\' min=\'0\' max=\'9\' step=\'1\' id=\'img_cp_png_default_quality\' name=\'img_cp_png_default_quality\' value=\'' . $this->_settings['img_cp_png_default_quality'] . '\' style=\'max-width:30vw;\' required ><div style=\'padding-left:0.5em;padding-right:0.5em;\'>9 - Current value: </div><div name=\'jcogs_dpqs\'>' . $this->_settings['img_cp_png_default_quality'] . '</div></div>'
                        )
                    )
                ),

                // ----------------------------------------
                // Set default background colour
                // ----------------------------------------

                array(
                    'title' => 'img_cp_default_bg_color',
                    'desc' => 'img_cp_default_bg_color_desc',
                    'fields' => array(
                        'img_cp_default_bg_color' => array(
                            'type'  => 'html',
                            'content' => '<input type=\'color\' id=\'img_cp_default_bg_color\' name=\'img_cp_default_bg_color\' value=\'' . $this->_settings['img_cp_default_bg_color'] . '\'>'
                        )
                    )
                )
            )
        );

        $sections[lang('jcogs_img_image_operational_defaults')] = array(
            'group' => 'img_options',
            'settings' => array(


                // ----------------------------------------
                // Set default image width
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_img_width',
                    'desc' => 'jcogs_img_cp_default_img_width_desc',
                    'group' => 'svg_options',
                    'fields' => array(
                        'img_cp_default_img_width' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_img_width'],
                            'required' => true
                        )
                    )
                ),

                // ----------------------------------------
                // Set default image height
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_img_height',
                    'desc' => 'jcogs_img_cp_default_img_height_desc',
                    'group' => 'svg_options',
                    'fields' => array(
                        'img_cp_default_img_height' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_img_height'],
                            'required' => true
                        )
                    )
                ),

                // --------------------------------------
                // Enable SVG passthrough option?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_svg_passthrough',
                    'desc' => 'jcogs_img_cp_enable_svg_passthrough_desc',
                    'fields' => array(
                        'img_cp_enable_svg_passthrough' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_svg_passthrough'],
                            'group_toggle' => array(
                                'y' => 'svg_options'
                            )
                        )
                    )
                ),

                // ------------------------------------------
                // Enable animated gif passthrough dominance?
                // ------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_ignore_save_type_for_animated_gifs',
                    'desc' => 'jcogs_img_cp_ignore_save_type_for_animated_gifs_desc',
                    'fields' => array(
                        'img_cp_ignore_save_type_for_animated_gifs' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_ignore_save_type_for_animated_gifs']
                        )
                    )
                ),

                // --------------------------------------
                // Set allow_scale_larger as default option?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_allow_scale_larger_default',
                    'desc' => 'jcogs_img_cp_allow_scale_larger_default_desc',
                    'fields' => array(
                        'img_cp_allow_scale_larger_default' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_allow_scale_larger_default'],
                        )
                    )
                ),


                // --------------------------------------
                // Enable auto_sharpen as default option?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_auto_sharpen',
                    'desc' => 'jcogs_img_cp_enable_auto_sharpen_desc',
                    'fields' => array(
                        'img_cp_enable_auto_sharpen' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_auto_sharpen'],
                        )
                    )
                ),


                // --------------------------------------
                // Enable lazy loading as default option?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_lazy_loading',
                    'desc' => 'jcogs_img_cp_enable_lazy_loading_desc',
                    'fields' => array(
                        'img_cp_enable_lazy_loading' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_lazy_loading'],
                            'group_toggle' => array(
                                'y' => 'jcogs_img_cp_lazy_loading_mode'
                            )
                        )
                    )
                ),

                // ----------------------------------------
                // Set default lazy loading mode
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_lazy_loading_mode',
                    'desc' => 'jcogs_img_cp_lazy_loading_mode_desc',
                    'group' => 'jcogs_img_cp_lazy_loading_mode',
                    'fields' => array(
                        'img_cp_lazy_loading_mode' => array(
                            'type' => 'select',
                            'choices' => array(
                                'lqip'  => lang('jcogs_img_cp_lazy_loading_mode_lqip'),
                                'dominant_color' => lang('jcogs_img_cp_lazy_loading_mode_dominant_color'),
                                'html5' => lang('jcogs_img_cp_lazy_loading_mode_html5'),
                            ),
                            'value' => $this->_settings['img_cp_lazy_loading_mode'],
                            'required' => true
                        )
                    )
                ),

                // -----------------------------------------------------
                // Enable progressive enhancement mode for lazy loading?
                // -----------------------------------------------------

                array(
                    'title' => 'jcogs_img_cp_lazy_progressive_enhancement',
                    'desc' => 'jcogs_img_cp_lazy_progressive_enhancement_desc',
                    'group' => 'jcogs_img_cp_lazy_loading_mode',
                    'fields' => array(
                        'img_cp_lazy_progressive_enhancement' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_lazy_progressive_enhancement']
                        )
                    )
                ),

                // --------------------------------------
                // Enable default fallback image?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_default_fallback_image',
                    'desc' => 'jcogs_img_cp_enable_default_fallback_image_desc',
                    'fields' => array(
                        'img_cp_enable_default_fallback_image' => array(
                            'type' => 'select',
                            'choices' => array(
                                'n'  => lang('jcogs_img_cp_no_fallback_image_option'),
                                'yc' => lang('jcogs_img_cp_local_fallback_colour_option'),
                                'yl' => lang('jcogs_img_cp_local_fallback_image_option'),
                                'yr' => lang('jcogs_img_cp_remote_fallback_image_option')
                            ),
                            'value' => $this->_settings['img_cp_enable_default_fallback_image'],
                            'group_toggle' => array(
                                'yc' => 'jcogs_img_cp_fallback_image_colour',
                                'yl' => 'jcogs_img_cp_fallback_image_local',
                                'yr' => 'jcogs_img_cp_fallback_image_remote'
                            )
                        )
                    )
                ),

                // ----------------------------------------
                // Set default fallback colour fill
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_fallback_image_colour',
                    'group' => 'jcogs_img_cp_fallback_image_colour',
                    'desc' => 'jcogs_img_cp_fallback_image_colour_desc',
                    'fields' => array(
                        'img_cp_fallback_image_colour' => array(
                            'type'  => 'html',
                            'content' => '<input type=\'color\' id=\'img_cp_fallback_image_colour\' name=\'img_cp_fallback_image_colour\' value=\'' . $this->_settings['img_cp_fallback_image_colour'] . '\'>'
                        )
                    )
                ),

                // ----------------------------------------
                // Set default fallback local image
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_fallback_image_local',
                    'group' => 'jcogs_img_cp_fallback_image_local',
                    'desc' => 'jcogs_img_cp_fallback_image_local_desc',
                    'fields' => array(
                        'img_cp_fallback_image_local' => [
                            'type' => 'html',
                            'value' => $this->_settings['img_cp_fallback_image_local'],
                            'required' => false,
                            'content' => ee()->file_field->dragAndDropField('img_cp_fallback_image_local', $this->_settings['img_cp_fallback_image_local'], 'all', 'image'),
                        ]
                    )
                ),

                // ----------------------------------------
                // Set default fallback remote image
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_fallback_image_remote',
                    'group' => 'jcogs_img_cp_fallback_image_remote',
                    'desc' => 'jcogs_img_cp_fallback_image_remote_desc',
                    'fields' => array(
                        'img_cp_fallback_image_remote' => array(
                            'type'  => 'text',
                            'maxlength' => 255,
                            'value' => $this->_settings['img_cp_fallback_image_remote'],
                            'preload' => lang('jcogs_img_cp_fallback_image_remote_required'),
                            'required' => true,
                        )
                    )
                ),
            )
        );

        $sections[lang('jcogs_img_system_options_limits')] = array(
            'group' => 'img_options',
            'settings' => array(

                // --------------------------------------
                // Enable image Auto-adjust?
                // --------------------------------------

                array(
                    'title' => 'jcogs_img_cp_enable_auto_adjust',
                    'desc' => 'jcogs_img_cp_enable_auto_adjust_desc',
                    'fields' => array(
                        'img_cp_enable_auto_adjust' => array(
                            'type'  => 'yes_no',
                            'value' => $this->_settings['img_cp_enable_auto_adjust'],
                            'group_toggle' => array(
                                'y' => 'jcogs_img_cp_auto_adjust_mode'
                            )
                        )
                    )
                ),

                // ----------------------------------------
                // Set maximum image dimensions
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_max_image_dimension',
                    'group' => 'jcogs_img_cp_auto_adjust_mode',
                    'desc' => 'jcogs_img_cp_default_max_image_dimension_desc',
                    'fields' => array(
                        'img_cp_default_max_image_dimension' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_max_image_dimension'],
                            'required' => true
                        )
                    )
                ),


                // ----------------------------------------
                // Set maximum image size
                // ----------------------------------------

                array(
                    'title' => 'jcogs_img_cp_default_max_image_size',
                    'desc' => 'jcogs_img_cp_default_max_image_size_desc',
                    'fields' => array(
                        'img_cp_default_max_image_size' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['img_cp_default_max_image_size'],
                            'required' => true
                        )
                    )
                ),
            )
        );

        $this->_data += array(
            'cp_page_title' => lang('jcogs_img_image_options'),
            'base_url' => ee('CP/URL', 'addons/settings/jcogs_img/image_defaults')->compile(),
            'save_btn_text' => sprintf(lang('btn_save'), lang('img_image_settings')),
            'save_btn_text_working' => lang('btn_saving'),
            'sections' => $sections
        );

        // Tell EE to load the custom javascript for the page
        ee()->cp->load_package_js('form_controls');

        return array(
            'heading'       => lang('img_image_settings'),
            'breadcrumb'    => array(
                ee('CP/URL', 'addons/settings/jcogs_img/image_defaults')->compile() => lang('img_image_settings')
            ),
            'body'          => ee('View')->make('ee:_shared/form')->render($this->_data),
        );
    }

    public function license()
    {
        $this->_build_sidebar();

        ee()->cp->add_js_script(array(
            'file' => array('cp/form_group'),
        ));

        // --------------------------------------
        // Validate and then save any changes
        // --------------------------------------
        if ($_POST) {

            // Validation
            $validator = ee('Validation')->make();

            // Define custom validation rules
            // ------------------------------

            // 1) Valid License Key Format
            // ---------------------------
            $validator->defineRule('valid_license_key_format', function ($key, $value, $parameters) {
                // Assume license is valid format if still obscured
                if (trim($value) == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    return true;
                }

                // So an actual license - regex only works if license pattern is correct.
                if (!preg_match("/^([a-z0-9]{8})-([a-z0-9]{4})-([a-z0-9]{4})-([a-z0-9]{4})-([a-z0-9]{12})$/", trim($value))) {
                    return 'jcogs_lic_cp_invalid_license_key_format';
                }
                return true;
            });

            // 2) Valid License
            // ----------------
            $validator->defineRule('valid_license', function ($key, $value, $parameters) {
                // Only check with licensing server if key is in right format 
                // and we have an email address
                // License key format is valid
                // Do we have a license key email to use?
                $this->license_key_email =  ee()->input->post('jcogs_license_key_email') ?: $this->_settings['jcogs_license_key_email'];
                if (!$this->license_key_email) {
                    return 'jcogs_lic_cp_missing_license_key_email';
                }
                // So go validate!
                // If we still have the obscured placeholder value we're OK
                if (trim($value) == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    return true;
                }
                // Otherwise go validate
                $this->license_status = ee('jcogs_img:Licensing')->license_status($value, $this->license_key_email, true);
                $this->_settings['jcogs_license_mode'] = $this->license_status->status;
                if ($this->license_status->message == 'jcogs_lic_cp_unable_to_reach_licensing_server') {
                    // But license server not reachable so continue with previous license status!
                    return 'jcogs_lic_cp_unable_to_reach_licensing_server';
                }
                if (str_contains($this->license_status->message, 'the license key given') > 0) {
                    // The email given is not valid for this license!
                    return 'jcogs_lic_cp_because_invalid_license_key_email';
                }
                if ($this->license_status->status == 'invalid' || $this->license_status->status == 'demo') {
                    // But license is not valid!
                    return 'jcogs_lic_cp_invalid_license';
                }
                return true;
            });

            // 3) Valid License Email
            // ----------------------
            $validator->defineRule('valid_license_email', function ($key, $value, $parameters) {
                $this->license_key_email =  ee()->input->post('jcogs_license_key_email');
                if (!$this->license_key_email || !filter_var($this->license_key_email, FILTER_VALIDATE_EMAIL)) {
                    // invalid emailaddress
                    return 'jcogs_lic_cp_invalid_license_key_email';
                }
                $this->license_key_email = $this->license_key_email ?: $this->_settings['jcogs_license_key_email'];
                // Only works if license is not invalid.
                $email_status = ee('jcogs_img:Licensing')->validate_license_email($this->license_key_email);
                if (!(is_object($email_status) && $email_status->status == 'valid')) {
                    return 'jcogs_lic_cp_invalid_license_key_email';
                }
                return true;
            });

            // 4) Licensed or Local
            // --------------------
            $validator->defineRule('license_not_invalid', function ($key, $value, $parameters) {
                // Only works if license is not invalid.
                if ($this->_settings['jcogs_license_mode'] == 'invalid') {
                    return 'jcogs_lic_cp_invalid_license';
                }
                return true;
            });

            // 5) Valid staging server
            // -----------------------
            $validator->defineRule('valid_staging_domain', function ($key, $value, $parameters) {
                // Find out if we can poll the licensing server... 
                if (ee('jcogs_img:Utilities')->allow_url_fopen_enabled() && ($this->_settings['jcogs_license_mode'] == 'magic' || $this->_settings['jcogs_license_mode'] == 'valid') && isset($value)) {
                    $path_to_check = count(parse_url($value)) > 1 ? parse_url($value)['host'] : parse_url($value)['path'];
                    $result = @file_get_contents('https://' . $path_to_check, false);
                    // Did it work?
                    if (isset($http_response_header) && strstr($http_response_header[0], '200')) {
                        return true;
                    }
                    return  'jcogs_lic_cp_invalid_staging_domain';
                }
                return true;
            });

            // Set validation rules
            // --------------------

            $validator->setRules(array(
                'jcogs_license_key_email'        => 'whenPresent[jcogs_license_key]|email|valid_license_email',
                'jcogs_license_key'              => 'valid_license_key_format|valid_license',
                'jcogs_staging_domain'           => 'whenPresent|valid_staging_domain',
            ));

            // Do the validation
            // -----------------
            $result = $validator->validate($_POST);

            if ($result->isValid()) {

                $fields = array();
                // Get all $_POST values, store them in array and save them
                // Use ee input library as it cleans up POST entries on loading
                foreach ($_POST as $key => $value) {
                    $fields[$key] = ee()->input->post($key);
                }

                $fields = array_merge($this->_settings, $fields);

                // Fix obscured license field if we need to
                if ($fields['jcogs_license_key'] == ee('jcogs_img:Utilities')->obscure_key($this->_settings['jcogs_license_key'])) {
                    $fields['jcogs_license_key'] = $this->_settings['jcogs_license_key'];
                }

                // Now save the settings values
                ee('jcogs_img:Settings')->save_settings($fields);

                // Pop up a save confirmation if all went well.
                ee('CP/Alert')->makeInline('shared-form')
                    ->asSuccess()
                    ->withTitle(lang('preferences_updated'))
                    ->addToBody(lang('preferences_updated_desc'))
                    ->defer();

                // Redraw page now
                ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img/license')->compile());
            } else {
                $this->_data['errors'] = $result;
                ee('CP/Alert')->makeInline('shared-form')
                    ->asIssue()
                    ->withTitle(lang('settings_save_error'))
                    ->addToBody(lang('settings_save_error_desc'))
                    ->now();
            }
        }

        // No post data, so just draw the page

        // --------------------------------------
        // Build the form into $sections array
        // --------------------------------------

        $sections[lang('jcogs_lic_license')] = array(
            'group' => 'license_options',
            'settings' => array(

                // ----------------------------------------
                // Enter License Key
                // ----------------------------------------

                ee('jcogs_img:Licensing')->mcp_license_key_entry(ee()->input->post('jcogs_license_key') ?: $this->_settings['jcogs_license_key'], ee()->input->post('jcogs_license_key_email') ?: $this->_settings['jcogs_license_key_email']),

                // ----------------------------------------
                // Enter License Key email address
                // ----------------------------------------

                array(
                    'title' => 'jcogs_lic_cp_license_key_email',
                    'desc' => 'jcogs_lic_cp_license_key_email_desc',
                    'fields' => array(
                        'jcogs_license_key_email' => array(
                            'type'  => 'text',
                            'value' => $this->_settings['jcogs_license_key_email'],
                            'placeholder' => lang('jcogs_lic_cp_license_key_email_placeholder'),
                            'required' => false
                        )
                    )
                ),

                // ----------------------------------------
                // Enter Staging Domain
                // ----------------------------------------

                ee('jcogs_img:Licensing')->mcp_staging_domain_entry($this->_settings['jcogs_license_mode']),

            )
        );

        // Check to see if allow_url_fopen is enabled
        if (!ee('jcogs_img:Utilities')->allow_url_fopen_enabled()) {
            ee('CP/Alert')->makeBanner('jcogs_img_cp_allow_url_fopen_disabled')
                ->asImportant()
                ->withTitle(lang('jcogs_img_cp_allow_url_fopen_disabled'))
                ->addToBody(lang('jcogs_img_cp_allow_url_fopen_disabled_desc'))
                ->defer();
        }

        $this->_data += array(
            'cp_page_title' => lang('jcogs_lic_register_license'),
            'base_url' => ee('CP/URL', 'addons/settings/jcogs_img/license')->compile(),
            'save_btn_text' => sprintf(lang('btn_save'), lang('jcogs_lic_save_button')),
            'save_btn_text_working' => lang('btn_saving'),
            'sections' => $sections
        );

        // Tell EE to load the custom javascript for the page
        ee()->cp->load_package_js('form_controls');

        return array(
            'heading'       => lang('jcogs_lic_register_license'),
            'breadcrumb'    => array(
                ee('CP/URL', 'addons/settings/jcogs_img/license')->compile() => lang('img_image_settings')
            ),
            'body'          => ee('View')->make('ee:_shared/form')->render($this->_data),
        );
    }

    /**
     * Build the navigation menu for the module
     */
    private function _build_sidebar()
    {
        $sidebar = ee('CP/Sidebar')->make();

        $sd_div = $sidebar->addHeader(lang('jcogs_img_nav_title'));
        $sd_div_list = $sd_div->addBasicList();
        $sd_div_list->addItem(lang('jcogs_img_cp_main_settings'), ee('CP/URL', 'addons/settings/jcogs_img'));
        $sd_div_list->addItem(lang('jcogs_img_caching_sidebar_label'), ee('CP/URL', 'addons/settings/jcogs_img/caching'));
        $sd_div_list->addItem(lang('img_image_settings'), ee('CP/URL', 'addons/settings/jcogs_img/image_defaults'));
        $sd_div_list->addItem(lang('jcogs_img_advanced_settings'), ee('CP/URL', 'addons/settings/jcogs_img/advanced_settings'));
        $sd_div_list->addItem(lang('jcogs_lic_license'), ee('CP/URL', 'addons/settings/jcogs_img/license'));
        $sd_div_list->addItem(lang('nav_support_page'),  ee()->cp->masked_url(ee('App')->get('jcogs_img')->get('docs_url')));
        if (ee('jcogs_img:Settings')::$settings['img_cp_enable_debugging'] === 'y') {
            $sd_debug = $sidebar->addHeader(lang('jcogs_img_debug_info'));
            $sd_debug_list = $sd_debug->addBasicList();
            $sd_debug_list->addItem(sprintf(lang('jcogs_img_version'), ee('Addon')->get('jcogs_img')->getInstalledVersion()));
            $sd_debug_list->addItem(sprintf(lang('jcogs_img_debug_php_version'),  PHP_VERSION));
            $sd_debug_list->addItem(sprintf(lang('jcogs_img_debug_ee_version'),  APP_VER));
            $sd_debug_list->addItem(sprintf(lang('jcogs_img_debug_async_support'),  function_exists('pcntl_async_signals') ? 'Available' : 'Not available'));
        }
    }

    /**
     * Utility function to validate a remote image path
     *
     * @param  string $image_path
     * @return boolean
     */
    private function _valid_remote_image(string $image_path)
    {
        // No point trying if we can't access remote files... 
        if (!ee('jcogs_img:Licensing')->allow_url_fopen_enabled()) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_allow_url_fopen_disabled'));
            return false;
        }

        // Try and get image from remote URL
        $the_file = @file_get_contents($image_path);
        if (!$the_file) {
            // unable to read a file from remote location
            return false;
        }
        // Got a file, is it an image?
        // Create a suitably random filename to prevent inter-process clashes
        $random_file = 'jcogs_img-' . time() . random_int(1, 999);
        $hash = hash('whirlpool', $random_file) . '.jpg';
        // Get base path
        $base_path = ee('jcogs_img:Utilities')->get_base_path();
        if (!$base_path) {
            // $basepath is invalid, so bale
            return false;
        }
        // Make sure temp path also ends in a /
        $temporary_image_path = $base_path . rtrim(ee('jcogs_img:Settings')::$settings['img_cp_default_cache_directory'], '/') . '/';
        // Make sure target directory is valid
        if (!ee('Filesystem')->exists($temporary_image_path)) {
            ee('Filesystem')->mkDir($temporary_image_path);
        }
        // Save file
        $file_save_status = file_put_contents($temporary_image_path . $hash, $the_file);
        if (!$file_save_status) {
            return false;
        }
        $is_valid_file = exif_imagetype($temporary_image_path . $hash);
        unlink($temporary_image_path . $hash);
        return $is_valid_file ? true : false;
    }

    /**
     * Utility function to get information about cache activity
     *
     * @return array
     */
    private function _get_image_cache_info()
    {

        // Work out cache performance information
        $return = [];
        $return['inception_date'] = '';
        $return['number_of_cache_fragments'] = '';
        $return['number_of_cache_hits'] = '';
        $return['cummulative_filesize'] = '';
        $return['cache_performance_desc'] = lang('jcogs_img_image_cache_is_empty');
        $return['cache_clear_button_desc'] = lang('jcogs_img_image_cache_is_empty');

        // Get information on currently cached images (if any) from cache_log
        $cache_log = ee('jcogs_img:Utilities')->cache_utility('get', JCOGS_IMG_CLASS . '/cache_log');

        // Do we have a cache key to work with?
        if ($cache_log !== false) {
            $return['caches_found'] = [];
            $return['inception_date'] = array_key_exists('inception_date', $cache_log) ? $cache_log['inception_date'] : 0;
            $return['number_of_cache_fragments'] = count($cache_log) - 1;
            $return['number_of_cache_hits'] = 0;
            $return['cummulative_filesize'] = 0;
            $return['cummulative_processing_time'] = 0;
            foreach ($cache_log as $cache_fragment) {
                if (is_array($cache_fragment)) {
                    $cache_fragment['count'] = array_key_exists('count', $cache_fragment) ? $cache_fragment['count'] : 0;
                    $return['number_of_cache_hits'] += $cache_fragment['count'];
                    $return['cummulative_filesize'] += $cache_fragment['count'] > 0 ? $cache_fragment['cummulative_size'] : 0;
                    $return['cummulative_processing_time'] += $cache_fragment['count'] > 0 && array_key_exists('cummulative_processing_time', $cache_fragment) ? $cache_fragment['cummulative_processing_time'] : 0;
                    $return['cummulative_processing_time'] += $cache_fragment['count'] > 0 && array_key_exists('cummulative_processing_time', $cache_fragment) ? $cache_fragment['cummulative_processing_time'] : 0;
                    if(array_key_exists('cache_dir', $cache_fragment)){
                        if (!array_key_exists($cache_fragment['cache_dir'], $return['caches_found'])) {
                            $return['caches_found'][$cache_fragment['cache_dir']] = 1;
                        } else {
                            $return['caches_found'][$cache_fragment['cache_dir']] += 1;
                        }
                    }
                }
            }
            $return['cache_performance_desc'] = [
                'desc' =>
                sprintf(
                    lang('jcogs_img_cp_cache_performance_desc_cache'),
                    $return['number_of_cache_fragments'],
                    count($return['caches_found']) > 1 ? sprintf(lang('jcogs_img_cp_cache_performance_desc_cache_many'), count($return['caches_found'])) : lang('jcogs_img_cp_cache_performance_desc_cache_single'),
                    $return['number_of_cache_hits'],
                    ee('jcogs_img:Utilities')->formatBytes($return['cummulative_filesize']),
                    ee('jcogs_img:Utilities')->date_difference_to_now($return['inception_date']),
                    round($return['cummulative_processing_time'], 2)
                ),
                'fields' => array(
                    'img_cp_ee5_empty_fields_hack_1' => array(
                        'type'  => 'hidden',
                        'value' => '',
                        'required' => false
                    )
                )
            ];
            $return['cache_clear_button_desc'] = [
                'title' => 'jcogs_img_cp_cache_clear',
                'desc' =>
                sprintf(lang('jcogs_img_cp_cache_clear_desc'), $return['number_of_cache_fragments']) . PHP_EOL .
                    sprintf(lang('jcogs_img_cp_cache_clear_button'), ee('CP/URL', 'addons/settings/jcogs_img/clear_image_cache'), ''),
                'fields' => array(
                    'img_cp_ee5_empty_fields_hack_2' => array(
                        'type'  => 'hidden',
                        'value' => '',
                        'required' => false
                    )
                )
            ];
        } else {
            $return['cache_performance_desc'] = [
                'desc' => lang('jcogs_img_cp_cache_performance_desc_no_cache'),
                'fields' => array(
                    'img_cp_ee5_empty_fields_hack_3' => array(
                        'type'  => 'hidden',
                        'value' => '',
                        'required' => false
                    )
                )
            ];
            $return['cache_clear_button_desc'] = [
                'title' => 'jcogs_img_cp_cache_clear',
                'desc' =>
                $return['number_of_cache_fragments'] != '' ?
                    sprintf(lang('jcogs_img_cp_cache_clear_desc'), $return['number_of_cache_fragments']) :
                    lang('jcogs_img_cp_cache_clear_desc_empty') . PHP_EOL .
                    sprintf(lang('jcogs_img_cp_cache_clear_button'), ee('CP/URL', 'addons/settings/jcogs_img/clear_image_cache'), 'disabled'),
                'fields' => array(
                    'img_cp_ee5_empty_fields_hack_2' => array(
                        'type'  => 'hidden',
                        'value' => '',
                        'required' => false
                    )
                )
            ];
        }
        return $return;
    }

    /**
     * Clear Image Cache
     *
     * @return void
     */
    public function clear_image_cache()
    {
        // Clear image cache
        if ($this->cache_path_ok) {
            $image_cache_info = $this->_get_image_cache_info();
            if (!empty($image_cache_info['caches_found'])) {
                foreach ($image_cache_info['caches_found'] as $cache_path => $cache_count) {
                    if (ee('Filesystem')->exists(ee('jcogs_img:Utilities')->path($cache_path))) {
                        ee('Filesystem')->emptyDir(ee('jcogs_img:Utilities')->path($cache_path));
                    }
                }
            }

            // Clear ee cache
            ee('jcogs_img:Utilities')->cache_utility('delete', JCOGS_IMG_CLASS . '/');

            // Add a banner to confirm clearance
            ee('CP/Alert')->makeBanner('jcogs_img_cp_cache_cleared')
                ->asSuccess()
                ->withTitle(lang('jcogs_img_cp_cache_cleared'))
                ->addToBody(sprintf(lang('jcogs_img_cp_cache_cleared_desc'), $image_cache_info['number_of_cache_fragments']))
                ->defer();

            ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img/caching'), 'refresh');
        }
    }

    /**
     * Manual Image Cache Audit
     *
     * @return void
     */
    public function manual_cache_audit()
    {
        // Run a cache audit
        if ($this->cache_path_ok) {
            $cache_audit_results = ee('jcogs_img:ImageUtilities')->cache_audit(true);

            if ($cache_audit_results) {
                // Add a banner to confirm clearance
                ee('CP/Alert')->makeBanner('jcogs_img_cp_cache_audit_now_completed')
                ->asSuccess()
                ->withTitle(lang('jcogs_img_cp_cache_audit_now_completed'))
                ->addToBody(sprintf(lang('jcogs_img_cp_cache_audit_now_completed_desc'),$cache_audit_results['locations'],$cache_audit_results['removed']))
                ->defer();
            } else {
                // Add a banner to that cache audit failed
                ee('CP/Alert')->makeBanner('jcogs_img_cp_cache_audit_failed')
                ->asWarning()
                ->withTitle(lang('jcogs_img_cp_cache_audit_failed'))
                ->addToBody(lang('jcogs_img_cp_cache_audit_failed_desc'))
                ->defer();
            }
        } else {
            // Add banner reporting dodgy cache path
            ee('CP/Alert')->makeBanner('jcogs_img_cp_cache_path_not_OK')
            ->asWarning()
            ->withTitle(lang('jcogs_img_cp_cache_path_not_OK'))
            ->addToBody(lang('jcogs_img_cp_cache_path_not_OK_desc'))
            ->defer();
        }
        ee()->functions->redirect(ee('CP/URL', 'addons/settings/jcogs_img/caching'));
    }
}
