<?php if (!defined('BASEPATH')) exit('No direct script access allowed');
/**
 * Extension
 * =========
 * Handles processing of EE hooks
 * 
 * CHANGELOG
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

require_once PATH_THIRD . "jcogs_img/vendor/autoload.php";
require_once PATH_THIRD . "jcogs_img/config.php";

class Jcogs_img_ext
{
    public $version            = JCOGS_IMG_VERSION;
    public $settings;

    /** 
     * Notice that for extensions you must include $settings 
     * as a parameter in the constructor
     */
    public function __construct($settings = [])
    {
        $this->settings = $settings;
    }

    public function activate_extension()
    {
        $data = [
            [
                'hook'      => 'template_post_parse', // required
                'method'    => 'template_post_parse', // required
                'priority'  => 10,
                'enabled'   => "y", // y/n
                'version'   => $this->version,
                'class'     => __CLASS__,
                'settings'  => serialize($this->settings),
            ],
            [
                'hook'      => 'after_file_save', // required
                'method'    => 'after_file_update', // required
                'priority'  => 10,
                'enabled'   => "y", // y/n
                'version'   => $this->version,
                'class'     => __CLASS__,
                'settings'  => serialize($this->settings),
            ],
            [
                'hook'      => 'after_file_delete', // required
                'method'    => 'after_file_update', // required
                'priority'  => 10,
                'enabled'   => "y", // y/n
                'version'   => $this->version,
                'class'     => __CLASS__,
                'settings'  => serialize($this->settings),
            ]
        ];
        foreach ($data as $hook) {
            ee()->db->insert('extensions', $hook);
        }
    }

    /**
     * Method for after_file_update hook
     * Looks to see if an image in JCOGS Image cache originated from this file
     * If found, the image in cache is deleted if auto-manage enabled and message sent to CP
     * If auto-manage not enabled, and image in cache then simply a warning message is flashed up in CP
     *
     * @param   object File object of modified file
     * @param   array  Parameters of the object
     * @param   array  Elements of file record modified during update
     * @return  mixed Return the object unchanged
     */
    public function after_file_update($file, $values)
    {
        // Is cache_auto_manage enabled?
        if (substr(strtolower(ee('jcogs_img:Settings')::$settings['img_cp_cache_auto_manage']), 0, 1) != 'y') {
            return;
        }
        
        // Do we have a file?
        if ($file) {
            // Get the filename for the image being saved
            $filename_array = pathinfo($values['file_name']);

            // Get the local image pathinfo
            $image_path_info = pathinfo(ee('jcogs_img:Utilities')->parseFiledir($values['upload_location_id']));

            // Add the local path to image 
            $filename_array['source_path'] = $image_path_info['dirname'];

            // Set up array to hold details of affected images 
            $affected_images = array();

            // Get a copy of the current cache log 
            $cache_log = ee('jcogs_img:Utilities')->cache_utility('get', JCOGS_IMG_CLASS . '/cache_log');

            // Scan log file to get list of unique cache locations
            $cache_locations = [];
            if ($cache_log) {
                foreach ($cache_log as $key => $value) {
                    if (is_array($value) && array_key_exists('cache_dir', $value)) {
                        array_key_exists($value['cache_dir'], $cache_locations) ? $cache_locations[$value['cache_dir']] += 1 : $cache_locations[$value['cache_dir']] = 1;
                    }
                }
            }

            // If we have a cache log scan it for the filename, if not assume we have just cleared cache... 
            if ($cache_log) {
                foreach ($cache_log as $cache_fragment => $array) {
                    // Ignore any entry that is not an array value
                    if (is_array($array) && array_key_exists('cache_dir', $array)) {
                        if(str_contains($cache_fragment,strtolower($filename_array['filename'])) !== false) {
                            // This fragment name contains basename of our modified image. 
                            // Now check that fragment is based on image from same source folder
                            if(str_contains($array['sourcepath'],$filename_array['source_path'].'/'.strtolower($filename_array['basename'])) !== false) {
                                // We've got a match so add it to list of affected images
                                $affected_images[$cache_fragment] = $array['sourcepath'];
                                if (array_key_exists('img_cp_cache_auto_manage',ee('jcogs_img:Settings')::$settings) && substr(strtolower(ee('jcogs_img:Settings')::$settings['img_cp_cache_auto_manage']), 0, 1) == 'y') {
                                    // If we have permission, delete the image and clear it from cache_log
                                    // Delete the cache image
                                    $this_file_path = rtrim(ee('jcogs_img:Utilities')->path($array['cache_dir'] . '/' . $cache_fragment), '/');
                                    if (file_exists($this_file_path)) {
                                        unlink($this_file_path);
                                    }
                                    // Remove from cache log
                                    if ($cache_log && isset($cache_log[$cache_fragment])) {
                                        unset($cache_log[$cache_fragment]);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // Now do a notification if we don't have cache-management permission but we have found affected images
            if (count($affected_images) > 0 && array_key_exists('img_cp_cache_auto_manage',ee('jcogs_img:Settings')::$settings) && substr(strtolower(ee('jcogs_img:Settings')::$settings['img_cp_cache_auto_manage']), 0, 1) != 'y') {
                ee('CP/Alert')->makeBanner('shared-form')
                ->asIssue()
                ->withTitle(lang('jcogs_img_cp_auto_manage_would_have_fired'))
                ->addToBody(sprintf(lang('jcogs_img_cp_auto_manage_would_have_fired_desc'), count($affected_images)))
                ->now();
            } elseif (count($affected_images) > 0) {
                // Update cache log with any changes
                ee('jcogs_img:Utilities')->cache_utility('save', JCOGS_IMG_CLASS . '/cache_log', $cache_log, 31536000);
            }
        }
        return $file; // whatever happened, return $file
    }


    /**
     * Method for cp_custom_menu hook
     * Adds fly-out menu to add-on if it is put into sidebar
     *
     * @param   object File object for current side-bar menu
     */
    public function cp_custom_menu($menu)
    {
        // Only works for EE6 and later
        if (version_compare(APP_VER, '6.0.0', '>=')) {
            $sub = $menu->addSubmenu('JCOGS Image');
            $sub->addItem(lang('fly_system_settings'), ee('CP/URL')->make('addons/settings/jcogs_img'));
            $sub->addItem(lang('fly_cache_settings'), ee('CP/URL')->make('addons/settings/jcogs_img/caching'));
            $sub->addItem(lang('fly_image_settings'), ee('CP/URL')->make('addons/settings/jcogs_img/image_defaults'));
            $sub->addItem(lang('fly_advanced_settings'), ee('CP/URL')->make('addons/settings/jcogs_img/advanced_settings'));
            $sub->addItem(lang('fly_clear_cache'), ee('CP/URL')->make('addons/settings/jcogs_img/clear_image_cache'));
        }
    }

    public function disable_extension()
    {
        ee()->db->where('class', __CLASS__);
        ee()->db->delete('extensions');
        return true;
    }

    public function update_extension($current = '')
    {
        return true;
    }

    /**
     * Method for template_post_parse hook
     * Looks for a JCOGS Image Lazy class within template
     * If JCOGS Image Lazy class found, appends lazy javascript element
     *
     * @param   string  Parsed template string
     * @param   bool    Whether an embed or not
     * @param   integer Site ID
     * @return  string  Template string
     */
    public function template_post_parse($template, $sub, $site_id)
    {
        // is this the final template?
        if ($sub === false) {
            // if there are other extensions on this hook, get the output after their processing
            if (isset(ee()->extensions->last_call) && ee()->extensions->last_call) {
                $template = ee()->extensions->last_call;
            }

            // Look in $template for JCOGS Image Lazy Loading class
            preg_match('/data-ji-src/', $template, $matches, PREG_UNMATCHED_AS_NULL);
            if ($matches) {
                // insert the javascript stub to enable lazyloading
                $javascript = file_get_contents(PATH_THIRD . strtolower(JCOGS_IMG_CLASS) . '/javascript/lazy_load.min.js');
                $css = '<noscript><style>[data-ji-src]{display:none;}</style></noscript>';
                list($start, $rest) = explode('</head>', $template);
                list($middle, $end) = explode('</body>', $rest);
                $template = $start . PHP_EOL . $css . PHP_EOL . '</head>' . PHP_EOL . $middle . PHP_EOL. '<script>' . PHP_EOL . $javascript . PHP_EOL . '</script>' . PHP_EOL . '</body>' . PHP_EOL . $end;
            }
        }
        return $template; // whatever happened, return $template
    }
}
