<?php

/**
 * JCOGS Image Class
 * =================
 * Parent class for a JCOGS Image instance                       
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

namespace JCOGSDesign\Jcogs_img\Library;

require_once PATH_THIRD . "jcogs_img/config.php";

use Imagine\Gd\Imagine;
use Imagine\Image\ImageInterface;
use Imagine\Image\Box;
use Imagine\Image\PointSigned;
use Imagine\Image\Palette;
use Imagine\Filter;
use Contao\ImagineSvg\SvgBox;
use Maestroerror\HeicToJpg;
use ColorThief\ColorThief;
use JCOGSDesign\Jcogs_img\Filters as Filters;

class JcogsImage
{
    // Add some variables
    public $flags; // status flags used during operation
    public $ident; // holds information about filename / path etc.
    public $processed_image; // holds the processed image object if available
    public $output; // holds the output generated for this image
    public $params; // holds current validated tagparams
    private $settings; // holds a local copy of Image settings
    public $source_image_raw; // holds the raw original image if available
    public $source_image; // holds the original image object if available
    public $source_svg; // holds the original image object if available
    public $stats; // holds performance information for this image
    private $transformation; // utility instance of Imagine Transformation object
    public $var_prefix; // used if we are doing prefixed variables

    // Some image variables
    public $aspect_ratio;
    public $aspect_ratio_orig;
    public $faces;
    public $filesize;
    public $fill_color;
    public $filters;
    public $local_path;
    public $new_width;
    public $new_height;
    public $new_size;
    public $offset;
    public $offset_horizontal;
    public $offset_vertical;
    public $opacity;
    public $orig_filesize;
    public $orig_width;
    public $orig_height;
    public $orig_size;
    public $placeholder;
    public $position;
    public $repeat_offset_x;
    public $repeat_offset_y;
    public $repeats_x;
    public $repeats_y;
    public $return;
    public $return_url;
    public $rotation;
    public $save_path;
    public $tagdata;
    public $vars;

    function __construct()
    {
        // Initialise some variables
        $this->flags = new \stdClass();
        $this->settings = ee('jcogs_img:Settings')::$settings;
        $this->ident = new \stdClass();
        $this->params = new \stdClass();
        $this->output = new \stdClass();
        $this->stats = new \stdClass();
        $this->stats->transformation_count = 0;
        $this->transformation = new Filter\Transformation(new Imagine());
        $this->var_prefix = '';

        // Start a timer
        $this->stats->start_time = microtime(true);
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_object_processing_starts'));

        // Get the parameters
        $this->params = ee('jcogs_img:ImageUtilities')->get_parameters();
        $this->ident->cache_path = '/' . $this->params->cache_dir . '/';

        // Set some flags... 
        $this->flags->animated_gif = false; // Image is not a gif till shown otherwise
        $this->flags->aspect_scaled_border = false; // Default condition is not to have symmetric borders
        $this->flags->masked_image = false; // Default is to assume image has not been masked
        $this->flags->srcset = false; // Image is not a srcset till shown otherwise
        $this->flags->svg = false; // Image is not an svg till shown otherwise
        $this->flags->use_colour_fill = false; // Default is not to use colour fill
        $this->flags->using_cache_copy = false; // Default is to assume nothing in cache
        $this->flags->valid_image = false; // Start off without a valid image
        $this->flags->allow_scale_larger = false; // Start off with assumption that we cannot scale larger

        // Find out if we are doing a crop
        $this->flags->its_a_crop = substr(strtolower($this->params->crop), 0, 1) != 'n';

        // Find out if we are doing lazy loading
        if (property_exists($this->params, 'lazy')) {
            $this->flags->doing_lazy_loading = ($this->params->lazy && substr(strtolower($this->params->lazy), 0, 1) != 'n') || (!$this->params->lazy && $this->settings['img_cp_enable_lazy_loading'] == 'y');
        }

        // See if we can enlarge image during transformation
        if (property_exists($this->params, 'allow_scale_larger')) {
            $this->flags->allow_scale_larger = substr(strtolower($this->params->allow_scale_larger), 0, 1) == 'y';
        }
    }

    /**
     * Add a border to an image
     *
     * @param  integer $count
     * @return bool
     */
    public function add_border(int $count)
    {

        if ($this->flags->using_cache_copy || $this->flags->masked_image || !$this->params->border) {
            // if using cache copy or it is a masked image or we have no border specified, return
            return true;
        }

        // Unpack parameters
        $border = $this->unpack_border_params();
        if (!$border) {
            // Something went wrong retrieving border parameters
            return false;
        }

        // Add border filter to transformation queue
        $this->transformation->add(new Filters\Box_border($border['width'], $border['colour']), $count);

        return true;
    }

    /**
     * Add border to image that has been masked using drawing methods
     *
     * @param  integer $count
     * @return bool
     */
    protected function add_masked_image_border(int $count)
    {
        if ($this->flags->using_cache_copy || !$this->params->border || !$this->flags->masked_image) {
            // if using cache copy or border param not set or not a masked image return
            return true;
        }

        // Unpack parameters
        $border = $this->unpack_border_params();
        if (!$border) {
            // Something went wrong retrieving border parameters
            return false;
        }

        // Add border filter to transformation queue
        $this->transformation->add(new Filters\Mask_border_drawn($border['width'], $border['colour']), $count);

        return true;
    }

    /**
     * Add rounded corners to image
     *
     * @return bool
     */
    public function add_rounded_corners()
    {

        if ($this->flags->using_cache_copy || !$this->params->rounded_corners) {
            // if using cache copy or we have no corners specified, return
            return true;
        }
        // 1) Unpack parameters
        $rounded_corners_params = explode('|', $this->params->rounded_corners);
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_adding_rounded_corners'), $rounded_corners_params);
        // CE Image reference for params - https://docs.causingeffect.com/expressionengine/ce-image/user-guide/parameters.html#rounded-corners
        // Parameter list:
        // location (all, tl, tr, bl, br)
        // dimension (int)
        // background colour
        // pipe repeat 
        // if duplicate values, later values overwrite earlier values

        // Initialise values for each corner
        $rounded_corner_working = array(
            'tl' => array(
                'x' => 0,
                'y' => 0,
                'radius' => 0
            ),
            'tr' => array(
                'x' => $this->new_width,
                'y' => 0,
                'radius' => 0
            ),
            'bl' => array(
                'x' => 0,
                'y' => $this->new_height,
                'radius' => 0
            ),
            'br' => array(
                'x' => $this->new_width,
                'y' => $this->new_height,
                'radius' => 0
            ),
        );
        // set a flag
        $need_to_do_corners = false;
        // load up paramters into working array
        foreach ($rounded_corners_params as $corner_param) {
            $params = explode(',', $corner_param);
            if (isset($params[0]) && in_array(strtolower($params[0]), ['tl', 'tr', 'bl', 'br', 'all'])) {
                $radius = isset($params[1]) ? ee('jcogs_img:ImageUtilities')->validate_dimension($params[1], $this->new_width) : 0;
                if ($radius != false) {
                    $need_to_do_corners = true;
                    if (strtolower($params[0]) == 'all') {
                        foreach ($rounded_corner_working as $corner => $data) {
                            $rounded_corner_working[$corner]['radius'] = $radius;
                        }
                    } else {
                        $rounded_corner_working[strtolower($params[0])]['radius'] = $radius;
                    }
                } else {
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_rounded_corners_invalid_radius'), $corner_param[1]);
                }
            } else {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_rounded_corners_unknown_option'), $corner_param[0]);
            }
        }

        // do we have to do anything?
        if (!$need_to_do_corners) {
            return true;
        }

        // Add rounded corner filter to transformation queue
        $this->transformation->add(new Filters\Rounded_corners($rounded_corner_working), $this->stats->transformation_count++);

        // See if we are doing borders
        $this->flags->masked_image = true;
        $this->add_masked_image_border($this->stats->transformation_count++);

        return true;
    }

    /**
     * Add a text-overlay to an image
     *
     * @return bool
     */
    public function add_text_overlay()
    {
        if ($this->flags->using_cache_copy || !$this->params->text) {
            // if using cache copy or we have no watermark specified, return
            return true;
        }

        // If we have some params run Text Overlay filter
        // ===========================================
        if ($this->params->text) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_adding_text_overlay'), $this->params->text);
            // Add text_overlay filter to transformation queue
            $this->transformation->add(new Filters\Text_overlay($this->params->text), $this->stats->transformation_count++);
        }
        return true;
    }

    /**
     * Add a watermark to an image
     *
     * @return bool
     */
    public function add_watermark()
    {
        if ($this->flags->using_cache_copy || !$this->params->watermark) {
            // if using cache copy or we have no watermark specified, return
            return true;
        }

        // If we have some params run Watermark filter
        // ===========================================
        $watermark_params_raw = explode('|', $this->params->watermark);
        if (is_array($watermark_params_raw)) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_adding_watermark'), $watermark_params_raw);
            // Add watermark filter to transformation queue
            $this->transformation->add(new Filters\Watermark($watermark_params_raw), $this->stats->transformation_count++);
        }
        return true;
    }

    /**
     * Utility function: Applies filters to an image
     * Filters in array $this->filters and are applied
     * in the sequence they appear in the array.
     *
     * @return bool
     */
    private function apply_filters()
    {
        // Do we have any filters specified?
        if ($this->flags->using_cache_copy || is_null($this->params->filter)) {
            return true;
        }

        // List of valid filter options
        // key=parameter name => value=[intervention method name, default value]
        $valid_filters = [
            'auto_sharpen' => ['auto_sharpen', null],
            'blur' => ['blur', 1],
            'brightness' => ['brightness', 0],
            'colorize' => ['colorize', 0, 0, 0],
            'contrast' => ['contrast', 0],
            'dominant_color' => ['dominant_color', 10],
            'dot' => ['dot', 6, '', 'circle', 1],
            'edgedetect' => ['edgedetect', null],
            'emboss' => ['emboss', null],
            'emboss_color' => ['emboss_color', null],
            'face_detect' => ['face_detect', 'y'],
            'gaussian_blur' => ['blur', 1],
            'grayscale' => ['grayscale', null],
            'greyscale' => ['grayscale', null],
            'invert' => ['negation', null],
            'lqip' => ['lqip', null],
            'mask' => ['mask', null],
            'mean_removal' => ['mean_removal', null],
            'negate' => ['negation', null],
            'noise' => ['noise', 30],
            'opacity' => ['opacity', 100],
            'pixelate' => ['pixelate', 0, false],
            'replace_colors' => ['replace_colors', null],
            'scatter' => ['scatter', 3],
            'selective_blur' => ['selective_blur', 1],
            'sepia' => ['sepia', 'fast'],
            'sharpen' => ['sharpen', null],
            'smooth' => ['smooth', 1],
            'sobel_edgify' => ['sobel_edgify', null],
        ];

        // Unpack the filters provided to an array called $this->filters
        // of form $key = filter name, $value = option string.
        $filters_from_tag = explode('|', $this->params->filter);

        // Check each temp filter: if it is valid load default name and temp parameters into $this->filters as an array
        foreach ($filters_from_tag as $filter) {
            $this_filter = explode(',', $filter);
            // Get name from array (first value) - what is left are any settings for the filter
            $this_filter_name = array_shift($this_filter);
            if (array_key_exists($this_filter_name, $valid_filters)) {
                // Get filter and parameters
                $default_filter = $valid_filters[$this_filter_name];
                // Use correct name of filter for intervention
                $this_filter_name = array_shift($default_filter);
                // What's left in this_filter are the requested settings (check to ensure it is an array)
                $this_filter = is_array($this_filter) ? $this_filter : [];
                // What's left in default filter are the default settings (check to ensure it is an array)
                $default_filter = is_array($default_filter) ? $default_filter : [];
                // if there are filter settings required, merge given params with defaults
                // we cannot use array_merge here because the values in arrays have no keys
                if ($required_params = count($default_filter)) {
                    for ($i = 0; $i < $required_params; $i++) {
                        // If filter setting doesn't exist or is outside valid range use default
                        if (!isset($this_filter[$i])) {
                            $this_filter[$i] = $default_filter[$i];
                        }
                        // Noise - Values must be in range 0 -> 255
                        if (in_array($this_filter_name, ['noise'])) {
                            $this_filter[$i] = $this_filter[$i] > 255 ? 255 : $this_filter[$i];
                            $this_filter[$i] = $this_filter[$i] < 0 ? 0 : $this_filter[$i];
                        }
                        // Brightness / Colorize - Values must be in range -255 -> 255
                        if (in_array($this_filter_name, ['brightness', 'colorize'])) {
                            $this_filter[$i] = $this_filter[$i] > 255 ? 255 : $this_filter[$i];
                            $this_filter[$i] = $this_filter[$i] < -255 ? -255 : $this_filter[$i];
                        }
                        // Contrast - Values must be in range -100 -> 100, and inverted
                        if (in_array($this_filter_name, ['contrast'])) {
                            $this_filter[$i] = $this_filter[$i] > 100 ? 100 : $this_filter[$i];
                            $this_filter[$i] = $this_filter[$i] < -100 ? -100 : $this_filter[$i];
                            $this_filter[$i] = -$this_filter[$i];
                        }
                        // Blur / Opacity - values must be in range 0->100
                        if (in_array($this_filter_name, ['opacity', 'blur'])) {
                            $this_filter[$i] = $this_filter[$i] > 100 ? 100 : $this_filter[$i];
                            $this_filter[$i] = $this_filter[$i] < 0 ? 0 : $this_filter[$i];
                        }
                        // Sharpen - values must be in range 0->500
                        if (in_array($this_filter_name, ['sharpen'])) {
                            $this_filter[$i] = $this_filter[$i] > 500 ? 500 : $this_filter[$i];
                            $this_filter[$i] = $this_filter[$i] < 0 ? 0 : $this_filter[$i];
                        }
                        // Pixelate - values must be positive integers
                        if (in_array($this_filter_name, ['pixelate'])) {
                            $this_filter[$i] = $this_filter[$i] < 0 ? 0 : $this_filter[$i];
                        }
                        // Sepia - value must be 'fast' or 'slow'
                        if (in_array($this_filter_name, ['sepia'])) {
                            $this_filter[$i] = in_array($this_filter[$i], ['fast', 'slow']) ? $this_filter[$i] : $default_filter[$i];
                        }
                        // Check valid parameters for Scatter
                        if (in_array($this_filter_name, ['scatter'])) {
                            // We need filter[0] and filter[1] to be defined to proceed
                            // If filter[1] not specified set it to double the value for filter[0]
                            if (!isset($this_filter[0])) {
                                $this_filter[0] = $default_filter[0];
                            }
                            if (!isset($this_filter[1])) {
                                $this_filter[1] = $this_filter[0] * 2;
                            }
                            // filter 0 should be less than filter 1
                            if ($this_filter[0] < $this_filter[1]) {
                                // we are good to go... so bale
                                break;
                            } else {
                                // force filter[0] to be less than filter[1] - choose nearest integer to half value of filter[1] (round up)
                                $this_filter[0] = (int)round($this_filter[1] / 2, 0);
                                break;
                            }
                        }
                    }
                }
                // Write the adjusted values into filter stack
                $this->filters[$this_filter_name] = $this_filter;
            }
        }

        // Process the filters selected
        $this->applyFilters();

        return true;
    }

    /**
     * Applies image filters to the current object's image
     *
     * @return void
     */
    public function applyFilters()
    {
        if ($this->filters) {
            // Process the filters selected
            foreach ($this->filters as $filter => $filter_settings) {
                $these_settings = '';
                if (strlen(implode(',', $filter_settings)) > 0) {
                    $these_settings = lang('jcogs_img_filtering_image_params') . implode(',', $filter_settings);
                }
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_filtering_image_start'), $filter . $these_settings);
                if (in_array($filter, ['auto_sharpen', 'blur', 'brightness', 'colorize', 'contrast', 'dominant_color', 'dot', 'edgedetect', 'emboss', 'face_detect', 'grayscale', 'lqip', 'opacity', 'mask', 'mean_removal', 'negation', 'noise', 'pixelate', 'replace_colors', 'scatter', 'selective_blur', 'sepia', 'sharpen', 'smooth', 'sobel_edgify'])) {
                    // process local filters
                    switch ($filter) {
                        case 'auto_sharpen':
                            // Applies a variable degree of sharpening depending on amount of image size
                            // reduction applied during processing. This method uses intervention's built in
                            // sharpen filter - which 
                            // Sharpen array - new/orig width ratio -> sharpen value
                            $max_sharpen_values = [
                                '0.04' => 9,
                                '0.06' => 8,
                                '0.12' => 7,
                                '0.16' => 6,
                                '0.25' => 5,
                                '0.50' => 4,
                                '0.75' => 3,
                                '0.85' => 3,
                                '0.95' => 2,
                                '1.00' => 1,
                            ];
                            $sharpening_value = ee('jcogs_img:Utilities')->vlookup(max(min($this->new_width / $this->orig_width, 1), 0), $max_sharpen_values);
                            // If we get 'false' back from fast_nearest we couldn't get a match, so bale
                            if ($sharpening_value) {
                                // We didn't get 'false' so go ahead and apply the filter... 
                                // Also bracket value to be between 1 and 9 just in case...
                                $this->transformation->add(new Filters\Sharpen($sharpening_value), $this->stats->transformation_count++);
                            }
                            break;

                        case 'brightness':
                            // Add brightness filter to transformation queue
                            // Get brightness value
                            $brightness = isset($filter_settings[0]) ? $filter_settings[0] : 0;
                            // Scale brightness value to lie within 100/-100 range
                            $brightness = (int) round($brightness / 255 * 100, 0);
                            $this->transformation->add(new Filters\Brightness((int) $brightness), $this->stats->transformation_count++);
                            break;

                        case 'colorize':
                            // Add contrast filter to transformation queue
                            $this->transformation->add(new Filters\Colorize($filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'contrast':
                            // Add contrast filter to transformation queue
                            $this->transformation->add(new Filters\Contrast(...$filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'dominant_color':
                            // Add dominant color filter to transformation queue
                            $this->transformation->add(new Filters\Dominant_color(...$filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'dot':
                            // Convert colour string (if specified) to color object
                            $filter_settings[1] = $filter_settings[1] ? ee('jcogs_img:ImageUtilities')->validate_colour_string($filter_settings[1]) : null;
                            // Add dot half-tone filter filter to transformation queue
                            $this->transformation->add(new Filters\Dot_filter(...$filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'edgedetect':
                            // Add edgedetect filter to transformation queue
                            $this->transformation->add(new Filters\Edgedetect(), $this->stats->transformation_count++);
                            break;

                        case 'emboss':
                            // Add emboss filter to transformation queue
                            $this->transformation->add(new Filters\Emboss(), $this->stats->transformation_count++);
                            break;

                        case 'face_detect':
                            // Add face detection filter to transformation queue
                            // Use sensitvity value from within tag, otherwise use value of param.
                            $sensitivity = intval($this->unpack_param('face_detect_sensitivity'));
                            $show_rectangles = !isset($filter_settings[0]) || isset($filter_settings[0]) && strtolower(substr($filter_settings[0], 0, 1) == 'y') ? true : false;
                            $faces = !$this->faces || is_null($this->faces) ? [] : $this->faces;

                            $this->transformation->add(new Filters\Face_detect($sensitivity, $show_rectangles, $faces), $this->stats->transformation_count++);
                            break;

                        case 'lqip':
                            // Add lqip filter to transformation queue
                            $this->transformation->add(new Filters\Lqip(), $this->stats->transformation_count++);
                            break;

                        case 'grayscale':
                            // Add grayscale filter to transformation queue
                            $this->transformation->add(new Filters\Greyscale(), $this->stats->transformation_count++);
                            break;

                        case 'mask':
                            // Do we set transparency flag?
                            $this->flags->masked_image = in_array($this->params->save_as, ['png', 'webp']);

                            // Add mask filter to transformation queue
                            $this->transformation->add(new Filters\Mask($filter_settings), $this->stats->transformation_count++);

                            // See if we are doing borders
                            $this->add_masked_image_border($this->stats->transformation_count++);

                            break;

                        case 'mean_removal':
                            // Add mean_removal filter to transformation queue
                            $this->transformation->add(new Filters\Mean_removal(), $this->stats->transformation_count++);
                            break;

                        case 'negation':
                            // Add negation filter to transformation queue
                            $this->transformation->add(new Filter\Advanced\Negation(), $this->stats->transformation_count++);
                            break;

                        case 'noise':
                            // Add noise filter to transformation queue
                            $this->transformation->add(new Filters\Add_noise(intval($filter_settings[0])), $this->stats->transformation_count++);
                            break;

                        case 'opacity':
                            // Adjust the opacity of image
                            $given_opacity = isset($filter_settings[0]) ? abs(intval($filter_settings[0])) : 100;
                            $this->transformation->add(new Filters\Opacity($given_opacity), $this->stats->transformation_count++);
                            break;

                        case 'pixelate':
                            // Pixelate an image
                            $this->transformation->add(new Filters\Pixelate(...$filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'replace_colors':
                            // Replace one colour in an image with another
                            // Require min 2 parameters - from color, to colour, tolerance
                            $from_color = isset($filter_settings[0]) ? ee('jcogs_img:ImageUtilities')->validate_colour_string($filter_settings[0]) : null;
                            $to_color = isset($filter_settings[1]) ? ee('jcogs_img:ImageUtilities')->validate_colour_string($filter_settings[1]) : null;
                            $tolerance = isset($filter_settings[2]) ? $filter_settings[2] : 0;
                            if ($from_color && $to_color) {
                                $this->transformation->add(new Filters\Replace_colors($from_color, $to_color, $tolerance), $this->stats->transformation_count++);
                            }
                            break;

                        case 'scatter':
                            // Create a scattered version of an image
                            $this->transformation->add(new Filters\Scatter(...$filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'selective_blur':
                            // Apply the selective blur filter to the image
                            $this->transformation->add(new Filters\Selective_blur(), $this->stats->transformation_count++);
                            break;

                        case 'sepia':
                            // Two algorithms
                            // Two step process - shift to greyscale and then colorize
                            // From here: https://www.phpied.com/image-fun-with-php-part-2/
                            // Or pixel based method (the one used by CE Image it seems)
                            // From here: https://dyclassroom.com/image-processing-project/how-to-convert-a-color-image-into-sepia-image
                            if ($filter_settings[0] == 'fast') {
                                $this->transformation->add(new Filters\Sepia_fast(), $this->stats->transformation_count++);
                            } else {
                                $this->transformation->add(new Filters\Sepia_slow(), $this->stats->transformation_count++);
                            }
                            break;

                        case 'sharpen':
                            // uses unsharp mask rather than imagine filter
                            // Amount
                            $amount = isset($filter_settings[0]) && intval($filter_settings[0]) && $filter_settings[0] > 0 ? intval($filter_settings[0]) : 80;
                            // radius
                            $radius = isset($filter_settings[1]) && is_numeric($filter_settings[1]) && $filter_settings[1] > 0 ? floatval($filter_settings[1]) : 0.5;
                            // threshold
                            $threshold = isset($filter_settings[2]) && intval($filter_settings[2]) && $filter_settings[2] > 0 ? intval($filter_settings[2]) : 3;
                            // Add unsharp mask filter to transformation queue
                            $this->transformation->add(new Filters\Unsharp_mask($amount, $radius, $threshold), $this->stats->transformation_count++);
                            break;

                        case 'smooth':
                            // Smooth an image
                            $this->transformation->add(new Filters\Smooth(...$filter_settings), $this->stats->transformation_count++);
                            break;

                        case 'sobel_edgify':
                            // Get threshold value if one set
                            $threshold = isset($filter_settings[0]) && intval($filter_settings[0]) && $filter_settings[0] > 0 ? intval($filter_settings[0]) : 125;
                            $this->transformation->add(new Filters\Sobel($threshold), $this->stats->transformation_count++);
                            unset($threshold);
                    }
                } elseif (str_contains($filter, 'IMG_FILTER') !== false) {
                    // we've got a GD imagefilter filter
                    $this->transformation->add(new Filters\Gd\Apply_Gd_Filter($filter, $filter_settings), $this->stats->transformation_count++);
                } else {
                    // something bad happened... 
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagefilter_failed'), [$filter => $filter_settings]);
                }
            }
        }
    }

    /**
     * Utility function: Returns the filename to use for the image based on options chosen.
     *
     * @return bool
     */
    protected function build_filename()
    {
        // Get some file info and begin to build the filename we'll use to save processed image
        if (!$this->ident->orig_filename) {
            $this->ident->orig_filename = hash('tiger160,3', str_replace('%', 'pct', urlencode($this->params->src)));
        }

        // Just in case, urldecode the filename to get the one we will use for saving cached image
        $this->ident->orig_filename = urldecode($this->ident->orig_filename);

        // Just in case, if need be shorten base filename to something sensible but still unique
        if (strlen($this->ident->orig_filename) > $this->settings['img_cp_default_max_source_filename_length']) {
            $this->ident->orig_filename = trim(substr($this->ident->orig_filename, 0, $this->settings['img_cp_default_max_source_filename_length']) . random_int(1, 999));
        }

        // Encode it again just to be safe
        $this->ident->orig_filename = urlencode($this->ident->orig_filename);

        // If prefix / suffix or filename specified, update filename accordingly
        if (property_exists($this->params, 'filename') && $this->params->filename) {
            $this->ident->orig_filename = $this->params->filename;
        }

        // And remove % characters (Issue #284 - Ligtas Server wierdness)
        $this->ident->orig_filename = str_replace('%','_',$this->ident->orig_filename);

        if (property_exists($this->params, 'filename_prefix') && $this->params->filename_prefix) {
            $this->ident->orig_filename = $this->params->filename_prefix . $this->ident->orig_filename;
        }
        if (property_exists($this->params, 'filename_suffix') && $this->params->filename_suffix) {
            $this->ident->orig_filename = $this->ident->orig_filename . $this->params->filename_suffix;
        }

        $parameters_to_exclude_from_filename = ['add_dimensions', 'attributes', 'bulk_tag', 'cache', 'cache_dir', 'create_tag', 'cache_mode', 'cache', 'disable_browser_checks', 'encode_urls', 'exclude_regex', 'fallback_src', 'filename', 'filename_prefix', 'filename_suffix', 'hash_filename', 'image_path_prefix', 'lazy', 'output', 'overwrite_cache', 'palette_size', 'save_as', 'save_type', 'url_only', 'use_image_path_prefix'];
        // $new_filename = '';
        $options = array();
        // Build up string of all methods applied to image and their parameters
        foreach ($this->params as $param => $value) {
            if (!in_array($param, $parameters_to_exclude_from_filename)) {
                $options[$param] = $value;
                // $new_filename .= '_' . $param . '_' . $value;
            }
        }

        // Add an element to hash so we differentinated between valid and demo mode... 
        $options['license_mode'] = $this->settings['jcogs_license_mode'];

        // Build a hex equivalent of the cache time set for this image
        $cache_tag = is_numeric($this->params->cache) && $this->params->cache > -1 ? dechex($this->params->cache) : 'abcdef';

        // If requested, hash the filename
        if (property_exists($this->params, 'hash_filename') && strtolower(substr($this->params->hash_filename, 0, 1)) == 'y') {
            $this->ident->orig_filename = hash('tiger160,3', serialize($this->ident->orig_filename));
        }
        $options_string = serialize($options);
        // Hash the methods string
        $hash_filename = hash('tiger160,3', $options_string);

        // Return the completed filename
        $this->ident->output = $this->ident->orig_filename . $this->settings['img_cp_default_filename_separator'] . $cache_tag . $this->settings['img_cp_default_filename_separator'] . $hash_filename;

        return true;
    }

    /**
     * Work out if we have default fallback image option to apply
     *
     * @return bool
     */
    protected function evaluate_default_image_options()
    {
        // reset src parameter - if we get to hear we've been unable to retrieve what was there before
        $this->params->src = null;
        // set an activity marker
        $found_fallback_option = false;
        if ((strtolower(substr($this->settings['img_cp_enable_default_fallback_image'], 0, 1)) == 'y'
            && strtolower(substr($this->settings['img_cp_enable_default_fallback_image'], 1, 1)) == 'c'
        )) {
            // Option 1) A colour fill requested
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_image_supplied_using_colour_field_backup'), $this->settings['img_cp_fallback_image_colour']);
            // We only can do this if we have some idea how big to make the colour fill, so check that first.
            $width = $this->params->width ? $this->params->width : false;
            $width = !$width && $this->params->min_width ?  $this->params->min_width : false;
            $width = !$width && $this->params->min ?  $this->params->min : false;

            $height = $this->params->height ? $this->params->height : false;
            $height = !$height && $this->params->min_height ?  $this->params->min_height : false;
            $height = !$height && $this->params->min ?  $this->params->min : false;

            $aspect_ratio = $this->params->aspect_ratio ? $this->params->aspect_ratio : false;

            if (!$width && $height && $aspect_ratio) {
                $width = round($height / $aspect_ratio, 0);
            }

            if (!$height && $width && $aspect_ratio) {
                $height = round($width * $aspect_ratio, 0);
            }

            if (!$width) {
                $width = $this->settings['img_cp_default_img_width'];
            }

            if (!$height) {
                $height = $this->settings['img_cp_default_img_height'];
            }
            if ($width && $height) {
                $this->params->bg_color = ee('jcogs_img:ImageUtilities')->validate_colour_string($this->settings['img_cp_fallback_image_colour']);
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_image_supplied_using_colour_field_backup'), $this->settings['img_cp_fallback_image_colour']);
                // Give image fill some arbitary 'original' dimensions
                $this->orig_size = new Box($width, $height);
                $this->orig_width = $this->orig_size->getWidth();
                $this->orig_height = $this->orig_size->getHeight();
                $this->aspect_ratio_orig = $this->orig_height / $this->orig_width;
                $this->flags->allow_scale_larger = true; // We need to do this in case target image size is bigger than our default size.
                $file_info['filename'] = 'colour_field';
                $file_info['extension'] = 'jpg';
                $this->flags->use_colour_fill = true;
                $found_fallback_option = true;
            } else {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_not_enough_dimensions_for_colour_field'));
            }
        } elseif (
            strtolower(substr($this->settings['img_cp_enable_default_fallback_image'], 0, 1)) == 'y'
            && strtolower(substr($this->settings['img_cp_enable_default_fallback_image'], 1, 1)) == 'r'
        ) {
            // Option 2) Put in remote default backup image
            $this->params->src = $this->settings['img_cp_fallback_image_remote'];
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_image_supplied_using_remote_default'), $this->params->src);
            $found_fallback_option = true;
        } elseif (
            strtolower(substr($this->settings['img_cp_enable_default_fallback_image'], 0, 1)) == 'y'
            && strtolower(substr($this->settings['img_cp_enable_default_fallback_image'], 1, 1)) == 'l'
        ) {
            // Option 3) Put in local default backup image
            $this->params->src = ee('jcogs_img:Utilities')->parseFiledir($this->settings['img_cp_fallback_image_local']);
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_image_supplied_using_local_default'), $this->params->src);
            $found_fallback_option = true;
        }
        if (!$found_fallback_option) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_image_fallback_option'));
        }
        return true;
    }

    /**
     * Utility function: Generates 'lazy' images
     *
     * @param  [type] $type
     * @return string $this->placeholder->path (path to the lazy image file)
     */
    public function generate_lazy_placeholder_image($type = null)
    {
        // Get placeholder image as required (dominant colour or LQIP version)
        // If $type specified get one of those, else look in $this->params->lazy else lqip
        if (!$type) {
            $type = $this->params->lazy ?: 'lqip';
        } else {
            $type = in_array($type, ['lqip', 'dominant_color']) ? $type : 'lqip';
        }
        $this->params->lazy_type = $type;
        $this->params->lazy = $this->params->lazy != '' ? $this->params->lazy : $this->settings['img_cp_lazy_loading_mode'];

        // work out filename ... 
        $this->placeholder = new \stdClass;
        $this->placeholder->output = $this->ident->output . '_' . $type;
        $this->placeholder->save_path = rtrim(ee('jcogs_img:Utilities')->path($this->ident->cache_path . $this->placeholder->output . '.' . $this->params->save_as), '/');
        $this->placeholder->return_url = ee('jcogs_img:ImageUtilities')->get_image_path_prefix() . rtrim($this->ident->cache_path . $this->placeholder->output . '.' . $this->params->save_as, '/');

        // do we have a copy in cache?
        if (!ee('jcogs_img:ImageUtilities')->is_image_in_cache($this->placeholder->save_path)) {
            // not in cache so make one!

            // Need to clone the image object for this
            $copy_of_content = clone ($this);
            // is this image using cache copy?
            if ($copy_of_content->flags->using_cache_copy) {
                $copy_of_content->processed_image = (new Imagine())->open($copy_of_content->save_path);
                $copy_of_content->flags->using_cache_copy = false;
            }
            // If save_as not jpg then make a jpg copy too (for noscript tag)
            if ($copy_of_content->params->save_as != 'jpg' && !ee('jcogs_img:ImageUtilities')->is_image_in_cache('/' . $copy_of_content->ident->cache_path . $copy_of_content->ident->output . '.jpg')) {
                // First make a box with appropriate colour fill
                try {
                    $pe_image_size = $copy_of_content->processed_image->getSize();
                    $pe_image = (new Imagine())->create(new Box($pe_image_size->getWidth(), $pe_image_size->getHeight()), $this->params->bg_color);
                } catch (\Imagine\Exception\RuntimeException $e) {
                    // Creation of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    return FALSE;
                }
                // Second, overlay the original image
                try {
                    $pe_image->paste($copy_of_content->processed_image, new PointSigned(0, 0))->save(rtrim(ee('jcogs_img:Utilities')->path('/' . $copy_of_content->ident->cache_path . $copy_of_content->ident->output . '.jpg'), '/'));
                } catch (\Imagine\Exception\RuntimeException $e) {
                    // Creation of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    return FALSE;
                }
            }
            $copy_of_content->params->filter = $type;
            // update to a new transformation filter to clean out anything from original image
            $copy_of_content->transformation = new Filter\Transformation(new Imagine());
            $copy_of_content->apply_filters();
            $copy_of_content->processed_image = $copy_of_content->transformation->apply($copy_of_content->processed_image);
            // and save it
            $options = $type = 'lqip' ? ['quality' => 20] : ['quality' => $copy_of_content->params->quality];
            $copy_of_content->processed_image->save($this->placeholder->save_path, $options);
            // delete the working copy
            unset($copy_of_content);
        }
        return $this->placeholder->return_url;
    }

    /**
     * Utility function: Generate output to return to template
     *
     * @return array|bool
     */
    public function generate_output()
    {
        // Start a timer for this operation run
        $time_start = microtime(true);
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_generating_output'));

        // Add a pre-parse hook for quasi-compatibility with CE Image
        // This is not quite same as some variables are set after this point... but nearly!
        $this->tagdata = ee()->TMPL->tagdata;
        if (ee()->extensions->active_hook('jcogs_img_pre_parse')) {
            $this->tagdata = ee()->extensions->call('ce_img_pre_parse', $this->tagdata, $this->vars, null);
        }

        // Start by creating a container for the output we are building
        $this->return = '';

        // Are we returning relative or full paths?
        $the_output_url =  $this->vars[0][$this->var_prefix . 'made']; // default is to use relative paths
        if (strtolower(substr($this->settings['img_cp_class_always_output_full_urls'], 0, 1)) == 'y') {
            // full URLs requested
            $the_output_url =  $this->vars[0][$this->var_prefix . 'made_url'];
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_generating_full_urls'));
        }

        // What kind of output are we generating?
        // Options are: 
        //   1 - url only
        //   2 - single tag with output param -> parse of output param contents
        //   3 - tag data from a standard tag pair (i.e. not bulk tag pair) -> parse tagdata
        //   4 - create_tag set to 'n', no output param, no tagdata -> do nothing
        //   5 - create_tag set to 'y' or single tag with no create tag specified -> build tag

        // Option 1 - url_only
        if (substr(strtolower($this->params->url_only), 0, 1) == 'y') {
            $this->return = $the_output_url;

            // Option 2 - single tag + output parameter -> custom output
        } elseif ($this->params->bulk_tag == 'n' && $this->params->output != '') {
            $this->return = ee()->TMPL->parse_variables($this->params->output, $this->vars);

            // Option 3 - not bulk tag, there is tagdata and either no create_tag or create_tag == 'n' : just parse tagdata
        } elseif ($this->params->bulk_tag == 'n' && $this->tagdata && substr(strtolower($this->params->create_tag), 0, 1) != 'y') {
            $this->return = ee()->TMPL->parse_variables($this->tagdata, $this->vars);

            // Option 4 - create_tag = 'n' so do nothing...
        } elseif (substr(strtolower($this->params->create_tag), 0, 1) == 'n') {

            // Option 5 - create_tag == y, or single tag and no create_tag specified : build an img tag
        } else {
            // To build the tag we break the activities down into components:
            // 1) Adding dimension parameters if required
            // 2) Adding pass-through attributes class and style tags if there are any
            // 3) Reconfiguring src, srcset for lazy loading if required

            // 1) Adding dimensions parameters if required
            // -------------------------------------------
            $add_dims = false;
            // We have two different parameters for this, so need a bit of logic to work out ... 
            if ($this->params->add_dims) {
                // add_dims has priority if both are set
                $add_dims = substr(strtolower($this->params->add_dims), 0, 1) == 'y';
            } elseif ($this->params->add_dimensions) {
                $add_dims = substr(strtolower($this->params->add_dimensions), 0, 1) == 'y';
            }

            // if we have an animated gif then always add dimensions
            // $add_dims = $this->flags->svg || $this->flags->animated_gif ? true : $add_dims;
            $add_dims = $this->flags->animated_gif ? true : $add_dims;

            // add the dims if we are going to...
            // if we have a srcset term involved, do not add width
            $this->return .=  $add_dims && !$this->flags->srcset && $this->vars[0][$this->var_prefix . 'width'] > 0 ? ' width="' . $this->vars[0][$this->var_prefix . 'width'] . '"' : '';

            $this->return .=  $add_dims && $this->vars[0][$this->var_prefix . 'height'] > 0 ? ' height="' . $this->vars[0][$this->var_prefix . 'height'] . '"' : '';

            // 2) Adding attributes including any consolidated class / style tags.
            // -------------------------------------------------------------------
            if(substr(strtolower($this->settings['img_cp_attribute_variable_expansion_default']), 0, 1) == 'y') {
                $this->return .= ' ' . ee()->TMPL->parse_variables($this->vars[0][$this->var_prefix . 'attributes'], $this->vars);
            } else {
                $this->return .= ' ' . $this->vars[0][$this->var_prefix . 'attributes'];
            }

            // add tagdata if there is any... (unless we're processing a bulk tag)
            $this->return .= ' ' . substr(strtolower($this->params->bulk_tag), 0, 1) == 'n' && $this->tagdata ? preg_replace('/\{.*\}/', '', $this->tagdata) : '';

            // 3) If image is svg add role attribute to img tag (MDN - http://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#identifying_svg_as_an_image)

            if ($this->flags->svg) {
                $this->return .= ' role="img"';
            }

            // 4) Reconfiguring src, srcset for lazy loading if required
            // ---------------------------------------------------------
            $return = '';
            if (($this->flags->doing_lazy_loading && $this->settings['img_cp_lazy_loading_mode'] != 'html5') && substr(strtolower($this->params->lazy), 0, 1) != 'h' && !$this->flags->svg) {
                // We doing the full lazy option - lazy parameter not set to 'no' and not to 'html5'

                // 1) Build <img> tag
                if ($this->flags->srcset) {
                    $return = ' data-ji-srcset="' . $this->vars[0][$this->var_prefix . 'srcset_param'] . '" sizes="' . $this->vars[0][$this->var_prefix . 'sizes_param'] . '" ';
                }
                $return = '<img loading="lazy" src="' . $this->placeholder->return_url . '" data-ji-src="' . $the_output_url . '" ' . $return . $this->return . '>';

                // 2) If we are doing progressive enhancement add the noscript alternative tag
                if ($this->settings['img_cp_lazy_progressive_enhancement']) {
                    $return_ns = '';
                    $return .= '<noscript class="ji__progenhlazyns"><img src="' . ee('jcogs_img:ImageUtilities')->get_image_path_prefix() . $this->ident->cache_path . $this->ident->output . '.jpg' . '" ' . $return_ns . $this->return . '></noscript>';
                }
            } elseif (($this->params->lazy && substr(strtolower($this->params->lazy), 0, 1) == 'h') || (!$this->params->lazy && $this->settings['img_cp_enable_lazy_loading'] == 'y' && $this->settings['img_cp_lazy_loading_mode'] == 'html5') && !$this->flags->svg) {
                // We are doing html5 only lazy loading option
                if ($this->flags->srcset) {
                    $return = ' srcset="' . $this->vars[0][$this->var_prefix . 'srcset_param'] . '" sizes="' . $this->vars[0][$this->var_prefix . 'sizes_param'] . '" ';
                }
                $return = '<img loading="lazy" src="' . $the_output_url . '" ' . $return . $this->return . '>';
            } else {
                // We are not doing lazy loading
                if ($this->flags->srcset) {
                    $return = ' srcset="' . $this->vars[0][$this->var_prefix . 'srcset_param'] . '" sizes="' . $this->vars[0][$this->var_prefix . 'sizes_param'] . '" ';
                }
                // Get an image path - which depends on if we are doing prefixes
                $return = '<img src="' . $the_output_url . '" ' . $return . $this->return . '>';
            }
            $this->return = $return;
        }
        // Write to log
        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_generated_output'), microtime(true) - $time_start));
        return true;
    }

    /**
     * Utility function: Returns the value of the aspect ratio (e.g. Y=X*ratio) 
     * as a ratio based on inputs
     *
     * @param string $input
     * @return bool|float
     */
    private function get_aspect_ratio($input = '')
    {
        if (!$input) {
            return false;
        }
        if (!(stripos($input, '_', 1) || stripos($input, '/', 1) || stripos($input, ':', 1))) {
            // not a correctly formed ratio
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_invalid_aspect_ratio'), $input);
            return false;
        }
        preg_match('/^(\d*)(?:_|\/|\:)(\d*)/', $input, $matches, PREG_UNMATCHED_AS_NULL);
        if (is_int(intval($matches[2])) && is_int(intval($matches[1]))) {
            // we have stuff for a ratio
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_aspect_ratio_calc'), $input . ' => ' . $matches[2] / $matches[1]);
            return $matches[2] / $matches[1];
        }
        return false;
    }

    /**
     * Utility function: Get fit dimensions 
     * What are resize dimensions for the source image:
     * fit=cover: original image to reach edge of new shape on all sides
     * fit=contain: original image to be contained within new shape
     * fit not specified or fit=stretch: the image will get distored  (default)
     * 
     * Returns array with dimensions to allow for further adjustment before loading into image object
     *
     * @param  string $fit
     * @return array 
     */
    private function get_fit_dimensions(string $fit = 'cover')
    {
        $new_width = $this->new_width;
        $new_height = $this->new_height;

        // FIT Calcs
        // =========    

        // We avoid distortion by preserving original image ratio and resizing image to fit
        // in new bounding box specified (by new_width and new_height)

        if ($fit == 'contain' || $fit == 'cover') {
            // We have fit parameter specified

            // Are old and new aspect ratios not the same? If so need to work out what 
            // the right dimensions for new image are
            if ($this->aspect_ratio !=  $this->aspect_ratio_orig) {
                // aspect ratios are not same, so check for fit
                // Is new width * original aspect ratio (ar height) greater than specified height?
                if ((int)round($this->new_width * $this->aspect_ratio_orig, 0) > $this->new_height) {
                    // Yes (Y is bounding axis)
                    // Has the fit parameter been specified?
                    if ($fit == 'contain') {
                        // Fit="contain"  - y axis is new value
                        // apply aspect ratio to calculate new_width
                        $new_width = (int)round($this->new_height / $this->aspect_ratio_orig, 0);
                    } else {
                        // Fit="cover"
                        // new_width needs to stay unchanged to fit on X axis, so work out new_height by
                        // Y is bounding axis, so new_height unchanged and work out new_width by
                        // applying original aspect ratio to new_height
                        $new_height = (int)round($this->new_width * $this->aspect_ratio_orig, 0);
                    }
                } else {
                    // No (X is bounding axis)
                    // Has the fit parameter been specified?
                    if ($fit == 'contain') {
                        // Fit="contain"  - x axis is as in new value
                        // X is bounding axis, so new_width unchanged and work out new_height by
                        // applying original aspect ratio to new_width
                        $new_height = (int)round($this->new_width * $this->aspect_ratio_orig, 0);
                    } else {
                        // Fit="cover"
                        // new_height needs to stay unchanged to fit on Y axis, so work out new_width by
                        // applying original aspect ratio to new_height
                        $new_width = (int)round($this->new_height / $this->aspect_ratio_orig, 0);
                    }
                }
            }
        }
        return [$new_width, $new_height];
    }

    /**
     * Utility function: Update content with filename, output from src
     * 1) Build filename for current option - is it in cache?
     * 2) If not in cache can we get a copy from source? 
     *
     * @param  string|null $path_to_source
     * @return bool
     */
    protected function get_image_from_src(string $path_to_source = null)
    {
        ee()->load->helper('url');
        ee()->load->helper('string');

        if (!$path_to_source) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_image_path_not_supplied'));
            return false;
        }

        // Build filename for currrent source image
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_attempting_to_get'), $path_to_source);
        $parsed_url = parse_url($path_to_source);
        if(!array_key_exists('path',$parsed_url)) return false;
        $file_info = pathinfo($parsed_url['path']);
        $this->ident->orig_filename = $file_info['filename'];
        $this->build_filename($this->ident->orig_filename);

        // If 'save_type' set then set save_as to that (target) format, otherwise choose a target format based on img_cp_default_image_format setting.
        if ($this->params->save_type) {
            $this->params->save_as = $this->params->save_type;
        } else {
            if ($this->settings['img_cp_default_image_format'] == 'source' && isset($file_info['extension'])) {
                $this->params->save_as =  $file_info['extension'];
            } else {
                $this->params->save_as = $this->settings['img_cp_default_image_format'];
            }
        }

        // Now normalise the file format (by checking that target format is valid for 
        // this server and supported by destination browser
        // If format is not valid, and original image (extension) is webp, png or gif, choose png
        // to preserve transparency if present, otherwise choose jpg
        if (!ee('jcogs_img:ImageUtilities')->validate_server_image_format($this->params->save_as) || (property_exists($this->params, 'disable_browser_checks') && is_null($this->params->disable_browser_checks) && !ee('jcogs_img:ImageUtilities')->validate_browser_image_format($this->params->save_as))) {
            if (isset($file_info['extension']) && in_array($file_info['extension'], ['webp', 'png', 'gif'])) {
                $this->params->save_as = 'png';
            } else {
                $this->params->save_as = 'jpg';
            }
        }

        // Generate / save local and url paths for processed image
        $this->local_path = $this->ident->cache_path  . $this->ident->output . '.' . $this->params->save_as;
        $this->save_path = rtrim(ee('jcogs_img:Utilities')->path($this->local_path), '/');

        // Check to see if we have a cache copy and overwrite_cache option not set
        $this->flags->using_cache_copy = ee('jcogs_img:ImageUtilities')->is_image_in_cache($this->save_path);

        // If image is in cache, check to see if it is an SVG 
        if ($this->flags->using_cache_copy) {
            $this->flags->svg = (bool) (ee('jcogs_img:ImageUtilities')->detect_sanitize_svg(@file_get_contents($this->save_path)));
            return true;
        } else {
            // Image not in cache or cache copy not available (options or expired), so we'll need to validate the source file.

            // Validate image
            $this->validate_image($path_to_source);

            // validate_image returns false if image does not validate
            $this->flags->valid_image ?
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_image_validated'), $path_to_source) : ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_source_not_valid'), $path_to_source);

            return $this->flags->valid_image;
        }
    }

    /**
     * Calculates the required dimensions for the processed image
     *
     * @return bool
     */
    protected function get_new_image_dimensions()
    // Get the params
    // Create a new image object
    // Apply the appropriate modifications
    // Return the image to template

    {
        if ($this->flags->using_cache_copy) {
            // Don't reprocess if we are using the cache copy
            return true;
        }

        // Step 1 - sort out the required dimensions of the image after the transformations

        // Width / Height
        // Get the requested width / height values (if there are any)
        // Dimensions from the tag
        $this->new_width = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('width'),$this->orig_width);
        $this->new_height = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('height'),$this->orig_height);

        // Work out if the srcset parameter requires larger new_width - if allow scale larger set then we make width the largest value specified
        if ($this->flags->allow_scale_larger && $this->params->srcset && !$this->flags->svg) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_srcset_and_ASL'));
            $srcset = explode('|', $this->params->srcset);
            $new_width = 0;
            // Now build the srcset entries and images
            foreach ($srcset as $width) {
                $width = ee('jcogs_img:ImageUtilities')->validate_dimension($width,$this->orig_width);
                if ($new_width > $width) break;
                $new_width = $width > $this->new_width && $new_width > $this->new_width ? $width : $this->new_width;
            }

            // Adjust image height to match change in image width to preserve aspect ratio
            if ($new_width > 0) {
                $this->new_height = round($this->new_height * $new_width / $this->new_width,0);
                $this->new_width = $new_width;
            }
        }

        // Process min/max values to get final dimensions for processed image
        $this->image_min_max_calcs();

        // Is original image smaller than final size?
        // If it is, and allow_scale_larger is not set to 'yes' then use original image dimensions instead
        if (
            !$this->flags->allow_scale_larger &&
            (($this->new_width && $this->new_width > $this->orig_width) ||
                ($this->new_height && $this->new_height > $this->orig_height))
        ) {
            // Set the new dimensions to be the same as original dimensions
            $this->new_width = $this->orig_width;
            $this->new_height = $this->orig_height;
            $this->new_size = new Box($this->orig_width, $this->orig_height);

            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_exceeds_source'), ['orig_width' => $this->orig_width, 'orig_height' => $this->orig_height, 'new_width' => $this->new_width, 'new_height' => $this->new_height,]);
        }

        // ASPECT RATIO Calcs
        // ==================
        // If Aspect Ratio is specified we need to adjust future image dimensions accordingly

        // If we are doing colour fill then we need to set some original image dimensions to match new ones.
        if ($this->flags->use_colour_fill) {
            $this->orig_width = $this->new_width;
            $this->orig_height = $this->new_height;
        }

        // First get aspect ratio of original image
        if (!$this->aspect_ratio_orig) {
            $this->aspect_ratio_orig = $this->orig_height && $this->orig_width ? $this->orig_height / $this->orig_width : null;
        }
        // Now get the requested aspect ratio, or guess something ... 
        if (!$this->aspect_ratio = $this->get_aspect_ratio($this->params->aspect_ratio)) {
            // We did not get anything, so guess... 
            if ($this->new_width && $this->new_height) {
                // we have both new_width and new_height so impute from that
                $this->aspect_ratio = $this->new_height / $this->new_width;
            } else {
                $this->aspect_ratio = $this->aspect_ratio_orig;
            }
        }

        // If we need to work out other dimension using Aspect Ratio
        // =========================================================
        // If only one dimension specified, work out other from aspect ratio
        // If two dimensions specified, ignore aspect ratio value
        // If we have width - calculate height
        // If have height - calculate width      
        // If we don't have a width at this point, but do have an aspect ratio... use original width as basis 
        if($this->aspect_ratio && !$this->new_width && !$this->new_height) {
            $this->new_width = $this->orig_width;
        }                 
        $this->new_width = $this->new_width ?: round($this->new_height / $this->aspect_ratio, 0);
        $this->new_height = $this->new_height ?: round($this->new_width * $this->aspect_ratio, 0);

        // If we still have no new_width or new_height at this point, assume image will be same size afterwards as before
        if (!$this->new_width && !$this->new_height) {
            $this->new_width = $this->orig_width;
            $this->new_height = $this->orig_height;
        }

        // FIT Calcs
        // =========    

        // At this point, if we are doing a crop or stretch resize we have all the info we need.
        // Otherwise, we need to work how adjust respecting crop and fit parameters to ensure
        // original image is not distorted during transformation.
        // We avoid distortion by preserving original image ratio and resizing image to fit
        // in new bounding box specified (by new_width and new_height)
        // * fit=cover: original image to reach edge of new shape on all sides
        // * fit=contain: original image to be contained within new shape  (default)
        // * fit not specified or fit=stretch: the image will get distored

        if (!$this->flags->its_a_crop && $this->params->fit != 'distort') {
            // It's not a crop (so it is a resize!)
            list($this->new_width, $this->new_height) = $this->get_fit_dimensions($this->params->fit);
        }
        $this->new_size = new Box($this->new_width, $this->new_height);
        return true;
    }

    /**
     * Checks that image is not too large to work with
     * Adjusts image to be smaller if Auto-adjust for Oversized Source Images is active
     * Auto-adjust works either on a dimension limit, or on a size limit.
     * 
     * @return bool
     */
    private function _get_working_image()
    {
        if(empty($this->source_image_raw) || is_null($this->source_image_raw)) return false;
                
        // Create an image to work with... 
        try {
            $this->source_image = (new Imagine())->setMetadataReader(new \Imagine\Image\Metadata\ExifMetadataReader())->load($this->source_image_raw);
        } catch (\Imagine\Exception\RuntimeException $e) {
            // Creation of image failed.
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
            return FALSE;
        }

        // Check for and store EXIF Orientation data associated with this image
        $orientation = $this->source_image->metadata()->get('ifd0.Orientation');
        
        // Check to see if image has orientation metadata then apply whatever transformation is required first
        // Options are:
        // 1 = Horizontal (normal)
        // 2 = Mirror horizontal
        // 3 = Rotate 180
        // 4 = Mirror vertical
        // 5 = Mirror horizontal and rotate 270 CW
        // 6 = Rotate 90 CW
        // 7 = Mirror horizontal and rotate 90 CW
        // 8 = Rotate 270 CW
        // From https://exiftool.org/TagNames/EXIF.html

        if($orientation && $orientation != 1) {
            // We have rotation information, and it is not 'do nothing'
            switch ($orientation) {
                case '2':
                    // Mirror horizontal
                    $this->source_image->flipHorizontally;
                    break;
                case '3':
                    // Rotate 180
                    $this->source_image->rotate(180);
                    break;
                case '4':
                    // Mirror vertical
                    $this->source_image->flipVertically;
                    break;
                case '5':
                    // Mirror horizontal and rotate 270 CW
                    $this->source_image->flipHorizontally->rotate(270);
                    break;
                case '6':
                    // Rotate 90 CW
                    $this->source_image->rotate(90);
                    break;
                case '7':
                    // Mirror horizontal and rotate 90 CW
                    $this->source_image->flipHorizontally->rotate(90);
                    break;
                case '8':
                    // Rotate 270 CW
                    $this->source_image->rotate(270);
                    break;                    
                default:
                    break;
            }
        }

        // Get image dimensions
        $image_dimensions = $this->source_image->getSize();
        if (!$image_dimensions) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_getimagesize_error'));
            return false;
        }

        // Get image filesize
        $check_image_size = strlen($this->source_image->get('jpg'));

        // Check to see if "Auto-adjust for Oversized Source Images" option is enabled
        // and image is not either SVG nor animated gif
        if (!$this->flags->animated_gif && !$this->flags->svg && substr(strtolower($this->settings['img_cp_enable_auto_adjust']), 0, 1) == 'y') {

            // Start a timer
            $auto_adjust_time_start = microtime(true);

            // First check source image against any max image dimension
            if ($this->settings['img_cp_default_max_image_dimension'] > 0 && max($image_dimensions->getWidth(), $image_dimensions->getHeight()) > $this->settings['img_cp_default_max_image_dimension']) {

                // Bigger than max image dimension so try to rescale dimensions
                ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_auto_adjust_active_dimensions'), $this->settings['img_cp_default_max_image_dimension']));

                // Get the rescale ratio we need
                $rescale_ratio = $this->settings['img_cp_default_max_image_dimension'] / max($image_dimensions->getWidth(), $image_dimensions->getHeight());

                // Create a box with new dimensions needed
                try {
                    $rescaled_image_box = new Box(
                        (int) round($image_dimensions->getWidth() * $rescale_ratio, 0),
                        (int) round($image_dimensions->getHeight() * $rescale_ratio, 0)
                    );
                } catch (\Imagine\Exception\RuntimeException $e) {
                    // Resize of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_auto_adjust_active_dimensions_failed'), $this->settings['img_cp_default_max_image_dimension']));
                    return false;
                }

                // Now rescale image to adjusted sizee
                $this->source_image->resize($rescaled_image_box);

                // Now update $image_dimensions
                $image_dimensions = $this->source_image->getSize();

                // And update size of this reprocessed image ... 
                $check_image_size = strlen($this->source_image->get('jpg'));
            }    

            // Second check the image filesize

            // Now check to see if image is larger than limit
            if ($check_image_size && $check_image_size > $this->settings['img_cp_default_max_image_size'] * 1000000) {
                // Larger so we'll need to rescale (possibly again)
                ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_auto_adjust_active_size'), $this->settings['img_cp_default_max_image_size']));

                // We are too large, so rescale back to sensible size...?
                // Get the rescale ratio we need
                $rescale_ratio = $this->settings['img_cp_default_max_image_size'] * 1000000 / $check_image_size;

                // Create a box with new dimensions needed
                try {
                    $rescaled_image_box = new Box(
                        (int) round($image_dimensions->getWidth() * $rescale_ratio, 0),
                        (int) round($image_dimensions->getHeight() * $rescale_ratio, 0)
                    );
                } catch (\Imagine\Exception\RuntimeException $e) {
                    // Resize of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_auto_adjust_active_dimensions_failed'), $this->settings['img_cp_default_max_image_dimension']));
                    return false;
                }

                // OK so now we need to rescale image... 
                $this->source_image->resize($rescaled_image_box);

                // Now update $image_dimensions
                $image_dimensions = $this->source_image->getSize();

                // And update size of this reprocessed image ... 
                $check_image_size = strlen($this->source_image->get('jpg'));                
            }

            // If we still do not have anything bale
            if (empty($this->source_image) || is_null($this->source_image)) {
                return false;
            }

            // Get the finish time for processing
            $auto_adjust_elapsed_time = microtime(true) - $auto_adjust_time_start;

            // Send a status message if we did anything
            if(!empty($rescaled_image_box)) {
                ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_auto_adjust_active_success'), max($rescaled_image_box->getWidth(), $rescaled_image_box->getHeight()), $check_image_size / 1000000, $auto_adjust_elapsed_time));
            }
        } else {
            // No auto-adjust so simply bale out if image is too large... 
            if ($check_image_size > $this->settings['img_cp_default_max_image_size'] * 1000000) {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_too_large_to_process'), ee('jcogs_img:Utilities')->formatBytes($check_image_size));
                return false;
            }
        }
        // If we get here we've got a source-image so do some housekeeping before returning

        // Get reprocessed image dimensions ... 
        $this->orig_size = $image_dimensions;
        // Expand image dimensions
        $this->orig_width = $this->orig_size->getWidth();
        $this->orig_height = $this->orig_size->getHeight();
        $this->aspect_ratio_orig = $this->orig_height / $this->orig_width;
        $this->ident->orig_image_filesize = $check_image_size;
        $this->flags->svg = false;

        return true;
    }

    /**
     * Utility function: Crop image
     *
     * @return bool
     */
    protected function image_crop()
    {

        if ($this->flags->using_cache_copy || !$this->flags->its_a_crop || !$crop_params = $this->validate_crop_params($this->params->crop)) {
            // if using cache copy or we have no crop specified or no valid crop parameters specified, return
            return true;
        }

        // Put marker in for start of crop
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_cropping_image_start'), $this->params->crop);
        
        $source_width = $this->orig_width;
        $source_height = $this->orig_height;
        
        // Is it a smart-scale crop?
        // Ignore smart-scaling if we are doing a face_detect crop and found faces 
        if ($crop_params[3] == 'y' && !($crop_params[0] == 'f' && $this->faces)) {
            
            // For smart-scale, the source image is resized before crop so that its smallest
            // dimension equals the length of the longest dimension of the target image.
            
            list($source_width, $source_height) = $this->get_fit_dimensions('cover');

            // Put a marker in the debug log
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_cropping_image_smart_scale'), ['new width' => $source_width, 'new height' => $source_height]);
            
            // We can't wait until filter queue runs to do resize, so create a temporary queue and do it now
            $transformation = new Filter\Transformation(new Imagine());
            $transformation->add(
                new Filter\Basic\Resize(
                    new Box((int) $source_width, (int) $source_height)
                ),
                $this->stats->transformation_count++
            );
            // Apply the resize 
            $this->processed_image = $transformation->apply($this->processed_image);
        }
        
        // Make sure crop is smaller than source image... 
        if ($this->new_width > $source_width || $this->new_height > $source_height) {
            // Can't do crop that is bigger than source
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_crop_exceeds_source'), ['pre-crop width' => $source_width, 'target crop width' => $this->new_width, 'pre-crop height' => $source_height, 'target crop height' => $this->new_height]);
            return false;
        }
        
        // Check to see if we have face_detect to consider
        if ($crop_params[1][0] == 'face_detect' || $crop_params[1][1] == 'face_detect' || $crop_params[0] == 'f') {
            // At least one, so get face detect data
            $this->faces = is_null($this->faces) ? ee('jcogs_img:ImageUtilities')->face_detection($this->processed_image->getGdResource(), intval($this->unpack_param('face_detect_sensitivity'))) : $this->faces;
            
            // If it is a face_detect crop and we got faces then adjust crop dimensions
            if ($this->faces && $crop_params[0] == 'f') {
                // Set width and height to match face detect bounding box plus face_crop_margin
                $face_crop_margin = intval($this->unpack_param('face_crop_margin'));
                $this->new_width = $this->faces[0]['width'] + $face_crop_margin * 2;
                $this->new_height = $this->faces[0]['height'] + $face_crop_margin * 2;
            }
        }

        // Co-ordinates to place top-left corner of new image points against original image
        $x_dimension['left'] = 0;
        $x_dimension['center'] = round(($source_width - $this->new_width) / 2,0);
        $x_dimension['right'] = round($source_width - $this->new_width,0);
        $x_dimension['face_detect'] = $x_dimension['center'];
        $y_dimension['top'] = 0;
        $y_dimension['center'] = round(($source_height - $this->new_height) / 2,0);
        $y_dimension['bottom'] = $source_height - $this->new_height;
        $y_dimension['face_detect'] = $y_dimension['center'];
        
        // If we got something from face_detect use it to re-calculate positions
        if ($this->faces && count($this->faces) > 1) {
            $centre_face_x = round($this->faces[0]['x'] + $this->faces[0]['width']/2,0);
            $centre_face_y = round($this->faces[0]['y'] + $this->faces[0]['height']/2,0);
            $x_dimension['face_detect'] = round($centre_face_x - $this->new_width/2,0);
            $y_dimension['face_detect'] = round($centre_face_y - $this->new_height/2,0);
        }

        // Calculate offset based on position and offset values
        $offset_x = (int) $x_dimension[$crop_params[1][0]] + $crop_params[2][0];
        $offset_y = (int) $y_dimension[$crop_params[1][1]] + $crop_params[2][1];
        
        // Now check to see if crop top-left offset is still within image
        // If not use top / left boundary as limit
        $offset_x = max(0, $offset_x);
        $offset_y = max(0, $offset_y);

        // Now check to see if far edge of crop is still within image
        // If not, push crop shape up to far edge
        $offset_x = min($source_width - $this->new_width, $offset_x);
        $offset_y = min($source_height - $this->new_height, $offset_y);

        // start point is a Point object (XY)
        $crop_start_point = new PointSigned(max($offset_x,0), max($offset_y,0));

        // Box size is simply new_width / new_height
        $crop_size = new Box($this->new_width, $this->new_height);

        // Add the crop to the transformation queue
        $this->transformation->add(new Filter\Basic\Crop($crop_start_point, $crop_size), $this->stats->transformation_count++);

        // Now in case we have a filter using $faces, if we have done a type 1 face_crop then update $faces
        if ($this->faces && $crop_params[0] == 'f') {
            $this->faces[0]['width'] += $face_crop_margin * 2;
            $this->faces[0]['height'] += $face_crop_margin * 2;
            $x_adjust = max($this->faces[0]['x'] - $face_crop_margin, 0);
            $y_adjust = max($this->faces[0]['y'] - $face_crop_margin, 0);
            $this->faces[0]['x'] -= $face_crop_margin;
            $this->faces[0]['y'] -= $face_crop_margin;
            for ($i = 0; $i < count($this->faces); $i++) {
                $this->faces[$i]['x'] -= $x_adjust;
                $this->faces[$i]['y'] -= $y_adjust;
            }
        }

        // Now in case we have a filter using $faces, if we have done a type 2 face_crop then update $faces
        if ($this->faces && ($crop_params[1][0] == 'face_detect' || $crop_params[1][1] == 'face_detect')) {
            // Work out origin shift from faces to crop
            $origin_shift_faces_x = $source_width != $this->new_width ? round(($source_width - $this->new_width)/2 + ($x_dimension['face_detect']-$x_dimension['center']),0) : 0;
            $origin_shift_faces_y = $source_height != $this->new_height ? round(($source_height - $this->new_height)/2 + ($y_dimension['face_detect']-$y_dimension['center']),0) : 0;
            for ($i = 0; $i < count($this->faces); $i++) {
                $this->faces[$i]['x'] = max(0,$this->faces[$i]['x'] - $origin_shift_faces_x);
                $this->faces[$i]['y'] = max(0,$this->faces[$i]['y'] - $origin_shift_faces_y);
                $this->faces[$i]['width'] = $this->faces[$i]['x'] + $this->faces[$i]['width'] < $this->new_width ? $this->faces[$i]['width'] : $this->new_width - $this->faces[$i]['x'];
                $this->faces[$i]['height'] = $this->faces[$i]['y'] + $this->faces[$i]['height'] < $this->new_height ? $this->faces[$i]['height'] : $this->new_height - $this->faces[$i]['y'];
            }
        }

        return true;
    }

    /**
     * Utility function: Flip image
     *
     * @return bool
     */
    protected function image_flip()
    {
        if ($this->flags->using_cache_copy || !$this->params->flip) {
            // if using cache copy or we have no flip specified, return
            return true;
        }

        // Add flips to transformation queue
        if ($this->params->flip != '') {
            $flips = explode('|', $this->params->flip);
            foreach ($flips as $flip) {
                // make sure values are legit flip options
                if ($flip == 'h') {
                    $this->transformation->add(new Filter\Basic\FlipHorizontally(), $this->stats->transformation_count++);
                }
                if ($flip == 'v') {
                    $this->transformation->add(new Filter\Basic\FlipVertically(), $this->stats->transformation_count++);
                }
            }

            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_flipping_image'), $this->params->flip);
        }
        return true;
    }

    /**
     * Utility function: Determines if there is a binding min or max value
     * that applies to image: if there is it calculates the appropriate dimension
     * value, otherwise leaves value at null
     *
     * @return bool
     */
    protected function image_min_max_calcs()
    {

        // Set all our min-max variables to null
        $min = null;
        $max = null;
        $min_width = null;
        $max_width = null;
        $min_height = null;
        $max_height = null;

        // Get adjusted values for any dimension parameters we have been supplied
        if ($this->params->min_width) {
            $min_width = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('min_width', $this->params), $this->orig_width);
        }
        if ($this->params->max_width) {
            $max_width = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('max_width', $this->params), $this->orig_width);
        }
        if ($this->params->min_height) {
            $min_height = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('min_height', $this->params), $this->orig_height);
        }
        if ($this->params->max_height) {
            $max_height = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('max_height', $this->params), $this->orig_height);
        }
        if ($this->params->max) {
            $max = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('max', $this->params), $this->orig_width);
        }
        if ($this->params->min) {
            $min = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('min', $this->params), $this->orig_width);
        }

        // Determine which max value is active
        $active_max_width = $max_width ?: $max;
        $active_max_height = $max_height ?: $max;

        // Determine which min value is active
        $active_min_width = $min_width ?: $min;
        $active_min_height = $min_height ?: $min;

        // Determine if we have new dimensions to specify

        // Width - Max
        if ($active_max_width) {
            if ($this->new_width) {
                $this->new_width = $active_max_width ? min($this->new_width, $active_max_width) : null;
            } else {
                $this->new_width = $active_max_width ? min($this->orig_width, $active_max_width) : null;
            }
        }

        // Width - Min
        if ($active_min_width) {
            if ($this->new_width) {
                $this->new_width = $active_min_width ? max($this->new_width, $active_min_width) : null;
            } else {
                $this->new_width = $active_min_width ? max($this->orig_width, $active_min_width) : null;
            }
        }

        // Height - Max
        if ($active_max_height) {
            if ($this->new_height) {
                $this->new_height = $active_max_height ? min($this->new_height, $active_max_height) : null;
            } else {
                $this->new_height = $active_max_height ? min($this->orig_height, $active_max_height) : null;
            }
        }

        // Height - Min
        if ($active_min_width) {
            if ($this->new_height) {
                $this->new_height = $active_min_height ? max($this->new_height, $active_min_height) : null;
            } else {
                $this->new_height = $active_min_height ? max($this->orig_height, $active_min_height) : null;
            }
        }
        return true;
    }


    /**
     * Utility function: Reflect image
     * Method loosely from p24 of https://www.slideshare.net/avalanche123/introduction-toimagine
     *
     * @return bool
     */
    protected function image_reflect()
    {

        if ($this->flags->using_cache_copy || is_null($this->params->reflection)) {
            // if using cache copy or we have no reflection specified, return
            return true;
        }

        // Default parameter values for reflection - '0,80,0,50%'
        if ($this->params->reflection != '') {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_reflecting_image'), (int) $this->params->reflection);
            // expand parameter
            $reflection_params = explode(',', $this->params->reflection);
            // validate the parameters obtained
            // first up gap (default 0)
            $reflection_params[0] = isset($reflection_params[0]) ? ee('jcogs_img:ImageUtilities')->validate_dimension($reflection_params[0], $this->new_height) : 0;
            // second up starting opacity (default 80)
            $reflection_params[1] = isset($reflection_params[1]) && intval($reflection_params[1]) != 0 && $reflection_params[1] > 0 && $reflection_params[1] <= 100 ? $reflection_params[1] : 80;
            // third up ending opacity (default 0)
            $reflection_params[2] = isset($reflection_params[2]) && intval($reflection_params[2]) != 0 && $reflection_params[2] > 0 && $reflection_params[2] <= 100 ? $reflection_params[2] : 0;
            // fourth up ending reflection height (default 50%)
            $reflection_params[3] = isset($reflection_params[3]) ? round(ee('jcogs_img:ImageUtilities')->validate_dimension($reflection_params[3], $this->new_height), 0) : round($this->new_height / 2, 0);
        } else {
            // If we get here somehow we got an empty parameter value into function - bale!
            return false;
        }

        // Work out background colour to use
        $reflection_colour = in_array($this->params->save_as, ['png', 'webp']) ? (new Palette\RGB())->color([0, 0, 0], 0) : $this->params->bg_color;

        // Add reflection filter to transformation queue
        $this->transformation->add(new Filters\Reflection($reflection_colour, $reflection_params[3], $reflection_params[0], $reflection_params[1], $reflection_params[2]), $this->stats->transformation_count++);

        unset($reflection_params);
        unset($reflection_colour);

        return true;
    }

    /**
     * Utility function: Resize image
     *
     * @return bool
     */
    protected function image_resize()
    {

        if ($this->flags->using_cache_copy || ($this->new_height == $this->orig_height && $this->new_width == $this->orig_width)) {
            // if using cache copy or if new size is identical to current size then no need to resize it, so return
            return true;
        }

        // Queue the resize...
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_resizing_image'), array($this->new_width, $this->new_height));
        $this->transformation->add(new Filter\Basic\Resize($this->new_size), $this->stats->transformation_count++);

        return true;
    }

    /**
     * Utility function: Rotate image
     *
     * @return bool
     */
    protected function image_rotate()
    {
        if ($this->flags->using_cache_copy || !$this->params->rotate) {
            // if using cache copy or it is a masked image or we have no border specified, return
            return true;
        }
        if ($this->params->rotate != '') {
            // make sure we got an integer rotation
            if (intval($this->params->rotate) == 0) {
                return true;
            }
            $this->transformation->add(new Filter\Basic\Rotate((int) $this->params->rotate, $this->params->bg_color), $this->stats->transformation_count++);
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_rotating_image'), (int) $this->params->rotate . ',' . $this->params->bg_color->__toString());
        }
        return true;
    }

    /**
     * Initialises a new JCOGS Image object
     * 1) Checks to see if we can get hold of the source image provided
     * 1.1) Build filename for current option - is it in cache?
     * 1.2) If not in cache can we get a copy from source? 
     *
     * @param  bool $use_fallback
     * @return bool
     */
    public function initialise($use_fallback = true)
    {

        // Establish what image we are going to use
        // ========================================
        // Work through each option (src, fallback_src, default options) and
        // use get_image_from_src to:
        // 1) Generate filename etc.
        // 2) See if filename points to something in cache - if in cache retrieve
        // 3) See if filename validates as an image - if so retrieve copy for processing

        // See if we have anything in the src parameter from tag... 
        if ($this->params->src != 'not_set' && array_key_exists('path',$parsed_url=parse_url($this->params->src)) && pathinfo($parsed_url['path'])) {
            // See if we can get an image with this
            if (!$this->get_image_from_src($this->params->src)) {
                if ($use_fallback) {
                    // Nothing on src, so try using fallback_src if there is one
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_src_image_supplied'));
                    // reset src parameter
                    // Try fallback_src if we have one... 
                    if ($this->params->fallback_src) {
                        if (!$this->get_image_from_src($this->params->fallback_src)) {
                            // If we get here we have no src or fallback_src 
                            // so now see if we have any default fallbacks... 
                            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_fallback_image_supplied'));
                        }
                    } else {
                        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_fallback_image_specified'));
                    }

                    // Have we got a valid image yet?
                    if (!$this->flags->valid_image) {
                        // Let's see if there are fallback image defaults provided?
                        $this->evaluate_default_image_options();
                        // Did we get something?
                        if (!$this->flags->use_colour_fill && !($this->params->src && $this->get_image_from_src($this->params->src))) {
                            return false;
                        }
                    }
                } else {
                    // Well we tried, if there is still no image here we can process, bale... 
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_backup_image_supplied'));
                    return false;
                }
            }
        } else {
            // No src specified so bale out 
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_src_image_supplied_2'));
            return false;
        }

        // If we get here, we have an image!
        // If image is an svg or gif and final output is also gif - skip rest of this step
        if (!($this->flags->svg || ($this->flags->animated_gif && ($this->params->save_as == 'gif' || $this->settings['img_cp_ignore_save_type_for_animated_gifs'] == 'y')))) {

            // If we get here, we are committed to losing the animated gif so remove flag accordingly
            $this->flags->animated_gif = false;

            // If we are not using the cache copy, create what will become our processed image
            if (!$this->flags->using_cache_copy) {
                // Add a background colour to main image if chosen image format doesn't 
                // support transparency, otherwise set one.
                $this->fill_color =
                    !in_array($this->params->save_as, array('png', 'gif', 'webp')) || $this->flags->use_colour_fill ?
                    $this->fill_color = $this->params->bg_color :
                    $this->source_image->palette()->color('000000', 0);

                // Now create the base for processed image
                // First make a box with appropriate colour fill
                try {
                    $this->processed_image = (new Imagine())->create($this->orig_size, $this->fill_color);
                } catch (\Imagine\Exception\RuntimeException $e) {
                    // Creation of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    return false;
                }

                // Second, unless we are building a colour-fill image, overlay the original image
                // This adds background to transparent images that are being saved in non-transparent formats
                if (!$this->flags->use_colour_fill) {
                    $this->processed_image->paste($this->source_image, new PointSigned(0, 0));
                }
            }
        }
        // All done!

        return true;
    }

    /**
     * Utility function: Post-process Image
     * Get image variable values for output if required
     * 
     * @return bool
     */
    public function post_process()
    {
        ee()->load->helper('url');

        // Start a timer for this operation run
        $time_start = microtime(true);
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_post_processing_image'));

        if (!$this->flags->use_colour_fill && !$this->flags->svg && !$this->flags->animated_gif) {
            // If we don't have it already (cache copy or other reason) set size and dimensions
            if ($this->save_path && ($this->flags->using_cache_copy || $this->flags->use_colour_fill)) {
                $this->filesize = (int) filesize($this->save_path);
                $new_image_dimensions = getimagesize($this->save_path);
                $this->new_width = array_key_exists(0,$new_image_dimensions) && $new_image_dimensions[0] > 0 ? (int) round($new_image_dimensions[0] + .4, 0) : null;
                $this->new_height = array_key_exists(1,$new_image_dimensions) && $new_image_dimensions[1] > 0 ? (int) round($new_image_dimensions[1] + .4, 0) : null;
            }

            // If we don't have it, get the image aspect ratio
            if (!property_exists($this, 'aspect_ratio') && $this->new_width > 0) {
                $this->aspect_ratio = $this->new_height / $this->new_width;
            }

            // If we don't have it, get the image file size
            if ($this->params->save_as && (!property_exists($this, 'filesize') || !$this->filesize)) {
                $this->filesize = strlen($this->processed_image->get($this->params->save_as));
            }

            // If we don't have it, get the image new_size value
            if ((!property_exists($this, 'new_size') || !$this->new_size) && $this->new_width > 0 && $this->new_height > 0) {
                $this->new_size = new Box($this->new_width, $this->new_height);
            }

            // Get the filesize of original image
            if ($this->flags->using_cache_copy) {
                $this->orig_filesize = '';
            } else {
                $this->orig_filesize = $this->ident->orig_image_filesize;
            }
        }

        if (!$this->flags->using_cache_copy) {
            if ($this->flags->svg) {
                // Fix up save as if we are processing an svg
                $this->params->save_as = 'svg';
            } elseif (!$this->flags->using_cache_copy && $this->flags->animated_gif) {
                // Fix up save as if we are processing a gif
                $this->params->save_as = 'gif';
            }
        }

        // Set mime type for processed image
        $this->ident->mime_type = ee('jcogs_img:ImageUtilities')->get_mime_type($this->params->save_as);

        // Prepare some defaults for the variable mapping
        $aspect_ratio_orig = property_exists($this, 'aspect_ratio_orig') && $this->aspect_ratio_orig ? $this->aspect_ratio_orig : '';
        $width_orig = property_exists($this, 'orig_width') && $this->orig_width ? (int) $this->orig_width : '';
        $height_orig = property_exists($this, 'orig_height') && $this->orig_height ? (int) $this->orig_height : '';
        if(!$this->flags->svg ) {
            $filesize = !$this->flags->use_colour_fill && property_exists($this, 'filesize') && $this->filesize ? $this->filesize : filesize($this->save_path);
        }
        $filesize_orig = !$this->flags->use_colour_fill && property_exists($this, 'orig_filesize') && $this->orig_filesize ? $this->orig_filesize : '';

        // Prepare the var prefix
        $var_prefix = '';

        $tag_parts = ee()->TMPL->tagparts;
        if (is_array($tag_parts) && isset($tag_parts[2])) {
            $var_prefix = $tag_parts[2] . ':';
            $this->var_prefix = $var_prefix;
        }

        // Assemble other variables for return
        // This set do not require any additional processing, so load them up since we already have them
        $this->vars[0] = [
            $var_prefix . 'aspect_ratio' => property_exists($this, 'aspect_ratio') && $this->aspect_ratio ? $this->aspect_ratio : $aspect_ratio_orig,
            $var_prefix . 'aspect_ratio_orig' => $aspect_ratio_orig,
            $var_prefix . 'attributes' => '',
            $var_prefix . 'made' => ee('jcogs_img:ImageUtilities')->get_image_path_prefix() . $this->ident->cache_path . $this->ident->output . '.' . $this->params->save_as,
            $var_prefix . 'made_with_prefix' => ee('jcogs_img:ImageUtilities')->get_image_path_prefix() . $this->ident->cache_path . $this->ident->output . '.' . $this->params->save_as,
            $var_prefix . 'orig' => $this->params->src == '' ?: parse_url($this->params->src)['path'],
            $var_prefix . 'made_url' => rtrim(base_url(), '/') . $this->ident->cache_path . $this->ident->output . '.' . $this->params->save_as,
            $var_prefix . 'orig_url' => (string) $this->params->src,
            $var_prefix . 'path' => property_exists($this, 'save_path') && $this->save_path ? $this->save_path : '',
            $var_prefix . 'path_orig' => $this->params->src != '' && ee('Filesystem')->exists(rtrim(ee('jcogs_img:Utilities')->path(parse_url($this->params->src)['path']), '/')) ? rtrim(ee('jcogs_img:Utilities')->path(parse_url($this->params->src)['path']), '/') : null,
            $var_prefix . 'width' => $this->new_width ?: $width_orig,
            $var_prefix . 'width_orig' => $width_orig,
            $var_prefix . 'height' => $this->new_height ?: $height_orig,
            $var_prefix . 'height_orig' => $height_orig,
            $var_prefix . 'name' => $this->ident->output,
            $var_prefix . 'name_orig' => property_exists($this->ident, 'orig_filename') ? $this->ident->orig_filename : '',
            $var_prefix . 'extension' => $this->params->save_as,
            $var_prefix . 'extension_orig' => property_exists($this->ident, 'orig_extension') ? $this->ident->orig_extension : '',
            $var_prefix . 'type' => $this->params->save_as,
            $var_prefix . 'type_orig' => property_exists($this->ident, 'orig_extension') ? $this->ident->orig_extension : '',
            $var_prefix . 'mime_type' => $this->ident->mime_type,
        ];

        // Do filesizes 
        if ($this->flags->use_colour_fill) {
            $this->vars[0][$var_prefix . 'filesize_bytes'] =  null;
            $this->vars[0][$var_prefix . 'filesize'] =  null;
        } else {
            if(!$this->flags->svg ) {
                $this->vars[0][$var_prefix . 'filesize_bytes'] =  ! $this->flags->svg && $filesize ? $filesize : filesize($this->save_path);
                $this->vars[0][$var_prefix . 'filesize'] =  ee('jcogs_img:Utilities')->formatBytes($this->vars[0][$var_prefix . 'filesize_bytes']);
            }
        }
        $this->vars[0][$var_prefix . 'filesize_bytes_orig'] = $filesize_orig;
        $this->vars[0][$var_prefix . 'filesize_orig'] = ee('jcogs_img:Utilities')->formatBytes($filesize_orig);


        // Set optional variable fields to null just in case ... 
        $this->vars[0][$var_prefix . 'lazy_image'] = '';
        $this->vars[0][$var_prefix . 'srcset_param'] = '';
        $this->vars[0][$var_prefix . 'sizes_param'] = '';

        // Now generate any variables that require additional processing

        // 1) Generate lazy image if required
        if (($this->flags->doing_lazy_loading && $this->settings['img_cp_lazy_loading_mode'] != 'html5') && substr(strtolower($this->params->lazy), 0, 1) != 'h' && !$this->flags->svg) {
            $this->vars[0][$var_prefix . 'lazy_image'] = $this->generate_lazy_placeholder_image();
            if (property_exists($this->params, 'lazy_type') && $this->params->lazy_type == 'dominant_color') {
                $this->vars[0][$var_prefix . 'dominant_color'] = $this->vars[0][$var_prefix . 'lazy_image'];
                $this->vars[0][$var_prefix . 'lqip'] = '';
            } else {
                $this->vars[0][$var_prefix . 'dominant_color'] = '';
                $this->vars[0][$var_prefix . 'lqip'] = $this->vars[0][$var_prefix . 'lazy_image'];
            }
        }

        // 2) Do we need to process tagdata?
        // If there is tagdata, or if there is an 'output' parameter specified
        // We also need to know if we need base64 or lqip/dominant_color images... 
        // so look specifically for these

        $haystack = ee()->TMPL->tagdata ?: '';
        $haystack .= $this->params->output;
        if (!$this->flags->svg) {
            if (stripos($haystack, '{' . $var_prefix . 'base64}') !== false) {
                if ($this->flags->using_cache_copy) {
                    $this->vars[0][$var_prefix . 'base64'] = ee('jcogs_img:ImageUtilities')->encode_base64((new Imagine())->open($this->save_path), isset(pathinfo($this->save_path)['extension']) ? pathinfo($this->save_path)['extension'] : 'png', $this->params->save_as == 'png' ? $this->params->png_quality : $this->params->quality);
                } else {
                    $this->vars[0][$var_prefix . 'base64'] = ee('jcogs_img:ImageUtilities')->encode_base64($this->processed_image, $this->params->save_as, $this->params->save_as == 'png' ? $this->params->png_quality : $this->params->quality);
                }
            }

            if (!$this->flags->using_cache_copy && stripos($haystack, '{' . $var_prefix . 'base64_orig}') !== false && property_exists($this, 'source_image_raw')) {
                $this->vars[0][$var_prefix . 'base64'] = ee('jcogs_img:ImageUtilities')->encode_base64((new Imagine())->load($this->source_image_raw), $this->params->save_as, $this->params->save_as == 'png' ? $this->params->png_quality : $this->params->quality);
            }

            if (stripos($haystack, '{' . $var_prefix . 'dominant_color}') !== false && $this->vars[0][$var_prefix . 'dominant_color'] = '') {
                $this->vars[0][$var_prefix . 'dominant_color'] = $this->generate_lazy_placeholder_image('dominant_color');
            }

            if (stripos($haystack, '{' . $var_prefix . 'average_color}') !== false) {
                // Get the GDImage object and run through colorthief
                $img = imagecreatefromstring($this->processed_image->__toString());
                $this->vars[0][$var_prefix . 'average_color'] = ColorThief::getColor($img, 10, null, 'hex');
                unset($img);
            }

            if (stripos($haystack, '{' . $var_prefix . 'aspect_ratio}') !== false && $this->vars[0][$var_prefix . 'aspect_ratio'] == '') {
                // Get here because it is a cache image - really want it so get aspect_ratio
                $ar_size = @getimagesize($this->save_path);
                $this->vars[0][$var_prefix . 'aspect_ratio'] = $ar_size ? $ar_size[2] / $ar_size[1] : '';
            }
        }

        // Add and consolidate class / style tags to rest.
        if ($this->params->consolidate_class_style == 'y' && ((property_exists($this, 'tagdata') && $this->tagdata) || ($this->params->attributes && trim($this->params->attributes) != ''))) {
            // if set in both attributes parameter and within an <> tag in tagdata, 
            // get class attributes
            $new_class = '';
            // get class attributes from parameter
            $class_array = $this->params->attributes && preg_match_all('/(?:class=)(\'|\")(.*?)\g1/', $this->params->attributes, $matches) ? $matches[2] : '';
            // consolidate class attributes (so we only get one of each)
            if (is_array($class_array)) {
                foreach ($class_array as $class) {
                    $new_class = !str_contains($new_class, $class) ? $new_class . ' ' . $class : $new_class;
                }
            }
            unset($class_array);
            // get class attributes from tagdata
            $class_array = $this->params->bulk_tag == 'n' && (property_exists($this, 'tagdata') && $this->tagdata) && preg_match_all('/(?:class=)(\'|\")(.*?)\g1/', $this->tagdata, $matches) ? $matches[2] : '';
            if (is_array($class_array)) {
                foreach ($class_array as $class) {
                    $new_class = !str_contains($new_class, $class) ? $new_class . ' ' . $class : $new_class;
                }
            }
            // get style attributes
            $new_style = $this->params->attributes && preg_match_all('/(?:style=)(\'|\")(.*?)\g1/', $this->params->attributes, $matches) ? implode(' ', $matches[2]) . ' ' : '';
            $new_style .= $this->params->bulk_tag == 'n' && (property_exists($this, 'tagdata') && $this->tagdata) && preg_match_all('/(?:style=)(\'|\")(.*?)\g1/', $this->tagdata, $matches) ? implode(' ', $matches[2]) : '';
            // remove class attributes from sources if required
            if ($this->params->attributes) {
                $this->params->attributes = preg_replace('/class=(\'|\").*?\g1/', '', $this->params->attributes);
                $this->params->attributes = preg_replace('/style=(\'|\").*?\g1/', '', $this->params->attributes);
            }
            if ($this->params->bulk_tag == 'n' && (property_exists($this, 'tagdata') && $this->tagdata)) {
                $this->tagdata = strlen($new_class) > 0 ? preg_replace('/class=(\'|\").*?\g1/', 'class="' . $new_class . '"', $this->tagdata) : $this->tagdata;
                $this->tagdata = strlen($new_style) > 0 ? preg_replace('/style=(\'|\").*?\g1/', 'style="' . $new_style . '"', $this->tagdata) : $this->tagdata;
            }
            // append the consolidated class / style info to the attributes parameter
            $this->params->attributes = !is_null($this->params->attributes) ? trim($this->params->attributes) : null;
            if ($new_class) {
                $this->params->attributes .= ' class="' . trim($new_class) . '"';
            }
            if ($new_style) {
                $this->params->attributes .= ' style="' . trim($new_style) . '"';
            }
        }
        // Fix up attributes variable
        $this->vars[0][$var_prefix . 'attributes'] = !is_null($this->params->attributes) ? trim($this->params->attributes) : null;

        // Work out the srcset images and parameter entries if required
        if ($this->params->srcset && !$this->flags->svg) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_srcset_begin'), $this->params->srcset);
            $srcset = explode('|', $this->params->srcset);
            // start building the srcset entry - we will have one (with just primary output) even if no values found
            $srcset_param = "";
            // we also need to build the sizes entry - we append image sizes as the media threshold to whatever is (or isn't) provided in the source tag
            $sizes_param = "";
            if ($this->params->sizes) {
                $sizes_param .= rtrim($this->params->sizes, ',') . ', ';
            }
            $current_entry = 0;

            // Now build the srcset entries and images
            foreach ($srcset as $width) {
                // srcset entries should be 
                // -> integer values 
                // -> greater than previous value 
                // -> less than width of original processed image
                $width = (int) ee('jcogs_img:ImageUtilities')->validate_dimension($width, $this->orig_width);
                if (is_numeric((int) $width) && $width > $current_entry && $width < $this->new_width) {
                    // append srscet width to filename
                    $srcset_filename = $this->ident->cache_path . $this->ident->output . '_' . $width . 'w.' . $this->params->save_as;
                    $srcset_filename_save_path = rtrim(ee('jcogs_img:Utilities')->path($this->ident->cache_path . $this->ident->output . '_' . $width . 'w.' . $this->params->save_as), '/');
                    // do we have a copy in cache?
                    if (!ee('jcogs_img:ImageUtilities')->is_image_in_cache(rtrim(ee('jcogs_img:Utilities')->path($srcset_filename), '/'))) {
                        // not in cache so make one
                        // generate image version
                        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_srcset_generate_image'), $srcset_filename);
                        if (!isset($srcset_image)) {
                            if (!$this->processed_image) {
                                $srcset_image = (new Imagine())->open($this->save_path);
                            } else {
                                $srcset_image = $this->processed_image->copy();
                            }
                        }
                        $srcset_size = $srcset_image->getSize();
                        // Widen image
                        $srcset_image->resize($srcset_size->widen($width));
                        // save srcset image version to cache
                        $srcset_image->save($srcset_filename_save_path, ['quality' => $this->params->quality]);
                        // delete the temporary image
                        unset($srcset_image);
                    }
                    // append srcset entry to list
                    $srcset_param .= ee('jcogs_img:ImageUtilities')->get_image_path_prefix() . $srcset_filename . ' ' . $width . 'w, ';
                    // append sizes condition for this image size to list
                    $sizes_param .= '(max-width:' . $width . 'px) ' . $width . 'px, ';
                    // increment $current_entry value
                    $current_entry = $width;
                }
            }
            // Append main image to srcset tag if we did any srcset work - otherwise skip
            if($current_entry > 0) {
                $next_entry = $current_entry + 1;
                $srcset_param .= $this->local_path . ' ' . $next_entry . 'w';
                // Append width one greater than final one used to sizes param
                $sizes_param .= $next_entry . 'px';

                // Write output to variables
                $this->vars[0][$var_prefix . 'srcset_param'] = $srcset_param;
                $this->vars[0][$var_prefix . 'sizes_param'] = $sizes_param;

                // Set a marker
                $this->flags->srcset = true;

                // delete the temporary image
                unset($srcset_image);
            } else {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_srcset_noop'));
            }
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_srcset_end'));
        }

        // Add local file meta data if its a local image
        // Clear the values just in case
        $this->vars[0][$var_prefix . 'img_title'] = '';
        $this->vars[0][$var_prefix . 'img_description'] = '';
        $this->vars[0][$var_prefix . 'img_credit'] = '';
        $this->vars[0][$var_prefix . 'img_location'] = '';

        // Get path to image 
        // Can only do it if we have original image so ... 
        if(property_exists($this->ident,'orig_image_path')) {
            $image_info = pathinfo(parse_url($this->ident->orig_image_path)['path']);
            if(strpos($image_info['dirname'],ee()->config->item('base_path')) == 0) {
                // path to image begins with EE base path 
                $local_path = str_replace(ee()->config->item('base_path'), '', $image_info['dirname']);
                // so check to see if local path is one that is an upload directory
                $destination = ee('Model')
                ->get('UploadDestination')
                ->with('Files')
                ->filter('server_path','LIKE','%'.$local_path.'%')
                ->filter('Files.file_name', $image_info['basename'])
                ->first();

                if($destination) {
                    // We found the file in the Files system
                    $file = $destination->Files->first();
                    $this->vars[0][$var_prefix . 'img_title'] = $file->title;
                    $this->vars[0][$var_prefix . 'img_description'] = $file->description;
                    $this->vars[0][$var_prefix . 'img_credit'] = $file->credit;
                    $this->vars[0][$var_prefix . 'img_location'] = $file->location;
                }
            }
        }
        // Write to log
        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_post_processed'), microtime(true) - $time_start));
        return true;
    }

    /**
     * Runs through sequence of actions to process an image
     *
     * @return bool
     */
    public function process_image()
    {

        // 1: Calculate dimensions of new image 
        // ====================================

        if (!$this->get_new_image_dimensions()) {
            return false;
        }

        // 2: Change size of processed image
        // =================================

        // Do a crop or resize if required?
        if (!$this->flags->svg && !$this->flags->animated_gif) {
            if ($this->flags->its_a_crop) {
                // It's a crop!
                if (!$this->image_crop()) {
                    return false;
                }
            } else {
                // It's a resize!
                if (!$this->image_resize()) {
                    return false;
                }
            }
        }

        // 3: Transform adjusted image
        // ===========================
        // Process sequence: flip, filters, text overlay, watermarks, rounded corners, borders, 
        // reflection, rotation

        // Flip image if such requested
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->image_flip()) {
            return false;
        }


        // Apply any filters Specified
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->apply_filters()) {
            return false;
        }

        // Add any text overlay requested
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->add_text_overlay()) {
            return false;
        }

        // Add any watermarking requested
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->add_watermark()) {
            return false;
        }

        // Apply rounded corners
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->add_rounded_corners()) {
            return false;
        }

        // Add any border if such requested
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->add_border($this->stats->transformation_count++)) {
            return false;
        }

        // Reflect image if such requested
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->image_reflect()) {
            return false;
        }

        // Rotate image if such requested
        if (!$this->flags->svg && !$this->flags->animated_gif && !$this->image_rotate()) {
            return false;
        }

        // Apply the transformation queue
        if(!$this->flags->svg && !$this->flags->animated_gif) {
            $temp_image = $this->transformation->apply($this->processed_image);
            if($temp_image) {
                $this->processed_image = $temp_image->copy();
                unset($temp_image);
            } else {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_filter_queue_failed'));
                return false;
            }
        }

        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_object_processing_ends'), microtime(true) - $this->stats->start_time));
        return true;
    }

    /**
     * Save the image
     *
     * @return bool
     */
    public function save()
    {
        // if we don't have a value for $this->save_path something has gone wrong...
        if (!isset($this->save_path) && $this->save_path) {
            if (property_exists($this->ident, 'output')) {
                // we were trying to produce something so report on what it was so we can debug
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_save_path_missing'), ['path($this->params->cache_dir)' => ee('jcogs_img:Utilities')->path($this->params->cache_dir), '$this->ident->output' => $this->ident->output, '$this->params->save_as' => $this->params->save_as, '$this->params->src' => $this->params->src]);
            }
            return false;
        }

        // Start a timer for this operation run
        $time_start = microtime(true);
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_saving_image_as'), $this->save_path);

        // Set image quality options
        $image_options = [];
        if (in_array($this->params->save_as, ['jpg', 'jpeg', 'webp'])) {
            $image_options['quality'] = (int) min(max($this->params->quality, 0), 100);
        } elseif (in_array($this->params->save_as, ['png'])) {
            // Set the png compression level
            $image_options['quality'] = (int) min(max($this->settings['img_cp_png_default_quality'], 0), 9);
        }

        // Set JPG Interlace if required
        if (!$this->flags->using_cache_copy && ($this->params->save_as == 'jpg' || $this->params->save_as == 'jpeg') && strtolower(substr($this->params->interlace, 0, 1)) == 'y') {
            $this->processed_image->interlace(ImageInterface::INTERLACE_LINE);
        }

        if ($this->flags->svg) {
            // If we are working with an svg image overwrite save_as parameter to svg
            $this->params->save_as = 'svg';
        } elseif ($this->flags->animated_gif) {
            // If we are working with a gif image overwrite save_as parameter to gif
            $this->params->save_as = 'gif';
        }

        // Make sure we set background colour in case we are mapping transparent image
        // to opaque format
        if (!in_array($this->params->save_as, ['gif', 'png', 'webp']) && !$this->flags->svg) {
            // Create an empty image the right size with correct background colour
            try {
                $temp_image = (new Imagine())->create($this->processed_image->getSize(), $this->params->bg_color);
            } catch (\Imagine\Exception\RuntimeException $e) {
                // Creation of image failed.
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                return FALSE;
            }
            // Paste image over top
            $temp_image->paste($this->processed_image, new PointSigned(0, 0));
            $this->processed_image = $temp_image->copy();
            unset($temp_image);
        }

        // If not an SVG or GIF image save to save path
        if (!$this->flags->svg && !$this->flags->animated_gif) {
            // We have try / catch here to trap errors caused when save path is not writeable
            try {
                $this->processed_image->save($this->save_path, $image_options);
            } catch (\Imagine\Exception\RuntimeException $e) {
                // Creation of image failed.
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                return FALSE;
            }

            // Add a saved hook for CE Img quasi-compatibility - for now only done for non-svg / gif files
            if (ee()->extensions->active_hook('jcogs_img_saved')) {
                ee()->extensions->call('jcogs_img_saved', $this->save_path, $this->params->save_as);
            }
        } elseif ($this->flags->svg) {
            // It is an svg so write sanitized input to output
            if (!file_put_contents(ee('jcogs_img:Utilities')->path($this->params->cache_dir) . $this->ident->output . '.svg', $this->source_svg)) {
                // Something went wrong
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_svg_save_failed'), rtrim(ee('jcogs_img:Utilities')->path($this->params->cache_dir) . $this->ident->output . '.' . $this->params->save_as, '/'));
            }
        } elseif ($this->flags->animated_gif) {
            // It is a gif so write input to output
            if (!file_put_contents(ee('jcogs_img:Utilities')->path($this->params->cache_dir) . $this->ident->output . '.gif', $this->source_image_raw)) {
                // Something went wrong
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_gif_save_failed'), rtrim(ee('jcogs_img:Utilities')->path($this->params->cache_dir) . $this->ident->output . '.' . $this->params->save_as, '/'));
            }
        }

        // Update cache log with record of this image.
        ee('jcogs_img:ImageUtilities')->update_cache_log($this->save_path, microtime(true) - $time_start, $this->params->cache_dir, property_exists($this->ident,'orig_image_path') ? $this->ident->orig_image_path : '');

        // Write to log
        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_saved'), microtime(true) - $time_start));
        return true;
    }

    /**
     * Unpacks the border parameters
     *
     * @return array|bool
     */
    protected function unpack_border_params()
    {
        if (!$this->params->border) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_border_param_issue'), $this->params->border);
            return false;
        }
        $border = array();
        $border_params = explode('|', $this->params->border);
        $border['width'] = ee('jcogs_img:ImageUtilities')->validate_dimension($border_params[0], $this->new_width);
        $border['colour'] = isset($border_params[1]) ? ee('jcogs_img:ImageUtilities')->validate_colour_string($border_params[1]) : $this->params->bg_color;

        return $border;
    }

    /**
     * Utility function: Returns the value of a parameter if one has been set, 
     * if not the default for that parameter is returned.
     *
     * @param string $param
     * @return mixed
     */
    protected function unpack_param(string $param)
    {
        if ($this->params->{$param}) {
            return $this->params->{$param};
        } else {
            return ee('jcogs_img:ImageUtilities')->get_parameters($param);
        }
    }


    /**
     * Utility function: Validates and unpacks crop setting (which has the form:
     * yes_or_no|position|offset|smart_scale) or installs defaults.
     *
     * @param string $param
     * @return mixed
     */
    protected function validate_crop_params(string $param)
    {
        // If we get null return false
        if (is_null($param)) {
            return false;
        }
        // Try to explode the param
        $crop_params = explode('|', $param);

        // Get the defaults
        $crop_defaults = explode('|', ee('jcogs_img:ImageUtilities')->get_parameters('crop', true));

        // Check we have something sensible returned... and replace with default if not
        // 1 - check crop param is yes / no / face_detect
        $crop_params[0] = in_array(strtolower(substr($crop_params[0], 0, 1)), ['y', 'n', 'f']) ? strtolower(substr($crop_params[0], 0, 1)) : $crop_defaults[0];
        // for params from [1] on we cannot be sure they were set when parameter was set by user... 
        // 2 - check position param is sensible (and expand)
        $crop_defaults[1] = explode(',', $crop_defaults[1]);
        $crop_params[1] = isset($crop_params[1]) && $crop_params[1] ? explode(',', $crop_params[1]) : $crop_defaults[1];
        $crop_params[1][0] = isset($crop_params[1][0]) && in_array(strtolower($crop_params[1][0]), ['left', 'center', 'right', 'face_detect']) ? strtolower($crop_params[1][0]) : $crop_defaults[1][0];
        $crop_params[1][1] = array_key_exists(1,$crop_params[1]) && isset($crop_params[1][0]) && in_array(strtolower($crop_params[1][1]), ['top', 'center', 'bottom', 'face_detect']) ? strtolower($crop_params[1][1]) : $crop_defaults[1][0];
        // 3 - check offset param (and expand)
        $crop_defaults[2] = explode(',', $crop_defaults[2]);
        $crop_params[2] = isset($crop_params[2]) && $crop_params[2] ? explode(',', $crop_params[2]) : $crop_defaults[2];
        $crop_params[2][0] = ee('jcogs_img:ImageUtilities')->validate_dimension($crop_params[2][0], $this->orig_width);
        $crop_params[2][1] = ee('jcogs_img:ImageUtilities')->validate_dimension($crop_params[2][1], $this->orig_width);
        // 4 - check smart_scale param
        $crop_params[3] = isset($crop_params[3]) && in_array(strtolower(substr($crop_params[3], 0, 1)), ['y', 'n']) ? strtolower(substr($crop_params[3], 0, 1)) : $crop_defaults[3];
        // 5 - check auto-center senstivity param
        $crop_params[4] = isset($crop_params[4]) && $crop_params[4] ? $crop_params[4] : $crop_defaults[4];

        // return
        return $crop_params;
    }

    /**
     * Utility function: Confirms that file link provided is:
     *  - a valid file
     *  - an image
     * 
     * If image is valid and not local, a copy is obtained
     * 
     * If an image is obtained a transient copy is returned
     * along with image dimensions / size info. 
     *
     * @param  string  $path_to_source
     * @param  bool $use_default
     * @return bool
     */

    /**
     * Utility function: Confirms that file link provided is:
     *  - a valid file
     *  - an image
     * 
     * If image is valid and not local, a copy is obtained
     * 
     * If an image is obtained a transient copy is returned
     * along with image dimensions / size info. 
     *
     * @param  string  $path_to_source
     * @param  bool $use_default
     * @return bool
     */
    public function
        validate_image(
        string $path_to_source,
        bool $use_default = true
    ) {
        // Save $this->ident->output/filename in temp location
        $local_image_output = property_exists($this->ident, 'output') ? $this->ident->output : null;
        $local_image_filename = property_exists($this->ident, 'filename') ? $this->ident->orig_filename : null;

        // Do we have anything to work with?
        if ($result = $this->source_image_raw = ee('jcogs_img:ImageUtilities')->get_a_local_copy_of_image($path_to_source)) {

            // If we get here we've got an image 
            // Do some preliminary processing of the image information
            $this->source_image_raw = $result['image_source'];
            $this->ident->orig_image_path = $result['path'];

            // Check to see if it is an HEIC image (which we can't process, so convert to jpg before we go any further... )
            if (ee('jcogs_img:ImageUtilities')->detect_heic($this->source_image_raw)) {
                try {
                    // Convert to jpeg
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_heic_conversion'));
                    $this->source_image_raw = HeicToJpg::convert($this->ident->orig_image_path)->get();
                } catch (\Exception $e) {
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_heic_error_1'), $e->getMessage());
                    try {
                        // Convert to jpeg
                        $this->source_image_raw = HeicToJpg::convertOnMac($this->ident->orig_image_path, "arm64")->get();
                    } catch (\Exception $e) {
                        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_heic_error_2'), $e->getMessage());
                        return FALSE;
                    }
                }
            }

            // Check to see if original image is a PNG ... 
            $this->flags->png = ee('jcogs_img:ImageUtilities')->detect_png_version($this->source_image_raw) ? true : false;

            // Check to see if original image is an animated GIF ... 
            // $local_image->orig_animated_gif = GifFrameExtractor::isAnimatedGif($path);
            $this->flags->animated_gif = ee('jcogs_img:ImageUtilities')->is_animated_gif($this->source_image_raw);

            // Check to see if original image is an SVG (use sanitizer)
            $this->source_svg = ee('jcogs_img:ImageUtilities')->detect_sanitize_svg($this->source_image_raw);
            $this->flags->svg = $this->source_svg ? true : false;

            // Get an image to work with
            if (!$this->flags->svg) {
                // Not an SVG, check and generate a suitable size for processing
                if(!$this->_get_working_image()) {
                    // Unable to resolve oversize so not valid image and bale
                    return false;
                }
            } else {
                // It's an SVG
                try {
                    $this->source_image = (new \Contao\ImagineSvg\Imagine())->load($this->source_svg);
                } catch (\Exception $e) {
                    // Creation of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    return FALSE;
                }
                // SVG Image so get dimensions from imagine-svg
                try {
                    $this->orig_size = $this->source_image->getSize();
                } catch (\Imagine\Exception\RuntimeException $e) {
                    // Creation of image failed.
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                    return FALSE;
                }

                // Fix up image dimensions if we have troublesome SVG ... 
                if (SvgBox::TYPE_NONE === $this->orig_size->getType()) {
                    // The image has no defined size so use default dimensions
                    $this->orig_width = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('default_img_width'));
                    $this->orig_height = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('default_img_height'));
                } elseif (SvgBox::TYPE_ASPECT_RATIO === $this->orig_size->getType()) {
                    // The image has a relative size, $size->getWidth() and $size->getHeight()
                    // should be treated as an aspect ratio
                    $this->aspect_ratio_orig = $this->orig_size->getHeight() / $this->orig_size->getWidth();
                    $this->orig_width = ee('jcogs_img:ImageUtilities')->validate_dimension($this->unpack_param('default_img_width'));
                    $this->orig_height = $this->aspect_ratio_orig * $this->orig_width;
                } else {
                    // The image has a defined size like a regular image
                    // $size->getType() would return SvgBox::TYPE_ABSOLUTE in this case
                    $this->aspect_ratio_orig = $this->orig_size->getHeight() / $this->orig_size->getWidth();
                    $this->orig_width = $this->orig_size->getWidth();
                    $this->orig_height = $this->orig_size->getHeight();
                }
                $this->flags->svg = true;
            }

            // Check that we have a filename for the image we retrieved
            $image_info = pathinfo(parse_url($this->ident->orig_image_path)['path']);
            if (!$image_info['filename']) {
                // No filename, so build one based on a hash of path, reversed to get elements most likely to differ
                // between (remote) files near front of hash process
                $image_info['filename'] = hash('tiger160,3', strrev(str_replace('%', 'pct', urlencode($this->ident->orig_image_path))));
            }

            // Save some information about the image
            foreach ($image_info as $item => $value) {
                $orig_item = 'orig_' . $item;
                $this->ident->{$orig_item} = $value;
            }

            $this->ident->orig_filename = $this->ident->orig_filename ?: $local_image_filename;
            $this->ident->output = $local_image_output;

            $this->flags->valid_image = true;
            unset($local_image_output);
            unset($local_image_filename);
            return true;
        } else {
            // still nothing so bale... 
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_local_copy_failed'), $local_image_filename ?? $this->params->src);
            $this->flags->valid_image = false;
            unset($local_image_output);
            unset($local_image_filename);
            unset($local_image);
            return false;
        }
    }
}
