<?php

/**
 * JCOGS Image Filter
 * ==================
 * Apply a mask filter
 * Takes two images - $image is the image to mask, $mask is the mask to apply
 * $mask is a black shape on a magic pink background.
 * 
 * @return object $image
 * 
 * CHANGELOG
 * 
 * 12/12/2022: 1.3      First release
 * 
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.3
 */

namespace JCOGSDesign\Jcogs_img\Filters\Gd;

use Imagine\Filter\FilterInterface;
use Imagine\Image\ImageInterface;
use Imagine\Image\Palette\Color\RGB;

/**
 * Apply a mask filter
 */
class Apply_mask implements FilterInterface
{
    /**
     * @var Imagine\Gd\Image
     */
    private $mask;

    /**
     * @var RGB
     */
    private $keep_color;

    /**
     * Apply a mask filter
     *
     * @param Imagine\Gd\Image $mask
     * @param RGB $color
     */
    public function __construct(ImageInterface $mask, RGB $keep_color = null)
    {
        $this->mask = $mask;
        $this->keep_color = $keep_color;
    }

    /**
     * {@inheritdoc}
     *
     * @see \Imagine\Filter\FilterInterface::apply()
     */
    public function apply(ImageInterface $image)
    {
        $size = $image->getSize();
        $maskSize = $this->mask->getSize();

        if ($size != $maskSize) {
            // throw new InvalidArgumentException(sprintf('The given mask doesn\'t match current image\'s size, Current mask\'s dimensions are %s, while image\'s dimensions are %s', $maskSize, $size));
        }

        // Flip image and mask into GDImage objects
        $working_image = imagecreatefromstring($image->__toString());
        imagealphablending($working_image, false); // Turn off alpha blending
        $working_mask = $this->mask->getGdResource();

        // Go through image and set transparency to match mask value at same point
        for ($x = 0, $width = $size->getWidth(); $x < $width; $x++) {
            for ($y = 0, $height = $size->getHeight(); $y < $height; $y++) {
                // Get the two colours at this point
                $mask_colour_at_point = imagecolorsforindex($working_mask, imagecolorat($working_mask, $x, $y));
                // Is mask colour magic pink?
                $new_alpha = $mask_colour_at_point['red'] == 255 && $mask_colour_at_point['green'] == 0 && $mask_colour_at_point['blue'] == 255;
                // If magic pink, update image pixel to be transparent
                if ($new_alpha == 127) {
                    $new_colour = imagecolorallocatealpha($working_image, 0, 0, 0, 127);
                    // Apply this colour to the working image
                    imagesetpixel($working_image, $x, $y, $new_colour);
                }
            }
        }

        // Convert back to Imagine image
        $image = ee('jcogs_img:ImageUtilities')->convert_GDImage_object_to_image($working_image);
        unset($working_image);

        return $image;
    }
}
